#pragma TextEncoding	 = "UTF-8"
#pragma IgorVersion		 = 8.00
#pragma rtGlobals		 = 3
#pragma rtFunctionErrors = 1
#pragma DefaultTab		 = {3, 20, 4}
#pragma moduleName		 = SuperQuickFit
#pragma version			 = 1.25

static Constant kVersion = 1.25
static StrConstant kVersionDate = "01/2026"

// --------------------- Project Updater header ----------------------
// If you're using Igor Pro 8 or later and have Project Updater
// installed, this package can check periodically for new releases.
// https://www.wavemetrics.com/project/Updater
static Constant kProjectID = 21866																		// the project node on IgorExchange
static StrConstant ksShortTitle = "Super Quick Fit"														// the project short title on IgorExchange

//__________________________________________________________________________________________________
//										Super Quick Fit
//	Written by Stephan Thuermer - https://www.wavemetrics.com/user/chozo
//
//	Provides a new right-click menu similar to the built-in Quick Fit. Several custom options have
//	been added for quick access, and user-defined functions are supported. To add your own function
//	you need to provide:
//		- a standard or all-at-once fit function with an unique name
//		- a guess function with the name fitfuncName + CoefGen_SUFFIX
//	
//	Optionally, derived results for any fit can be printed by providing a function named fitfuncName + derivedVal_SUFFIX
//
//	Both the guess and derived-results functions have the SuperQuickFitStruct as input.
//	The guess function is expected to generate a coefficient wave with the correct number of
//	parameters and return this wave to the s.cw parameter. Additionally, the s.holdStr can be set
//	to hold one parameter (usually the baseline) if the option is activated in the menu.
// 
//	Additional custom fit functions are provided with the package:
//		- Fermi-Dirac function with linear slope
//		- Line with x-crossing as input parameter (instead of the y-crossing)
//		- Angular distribution function (e.g., for Photoelectron Angular Distributions)
//		- Double Gauss function
//		- A Multi-Gauss function with selectable number of peaks.
//		- Exponentially Modified Gaussian (ExpModGauss or EMG)
//		- Convolution of two exponential decays (ExpConvExp)
//
//	Support for standard fit functions with the extended options is provided as well.
//
//	Error codes:
//	 -1 = no valid target or function
//	 -2 = no graph found
//	 -3 = designated graph does not exist
//	 -4 = trace not on graph
//	 -5 = invalid function type
//	 -6 = invalid contour trace
//	 -10 = guess for fit function failed
//	 -100 = image has separate x or y axis (not supported yet)
//	 -999 = execution of the fit failed
//
// 2022-04-07 - ver. 1.00:	Initial public release.
// 2022-04-08 - ver. 1.01:	Added textbox preferences.
// 2022-05-11 - ver. 1.02:	Fixed bug: Missing M_Covar for Line fit was not handled properly.
// 2022-07-14 - ver. 1.03:	Improved guess for the EF parameter in Fermi-Line guess function.
//							Improved derived parameters for the EMG peak shape.
// 2022-10-18 - ver. 1.04:	'Clean-up' now only removes fit-related variables.
//							Derived values will be added to wave notes as well.
//							Improved guess for DoubleGauss function.
// 2022-10-27 - ver. 1.05:	Added ExpConvExp function (thanks to Oleg Dyachok from Uppsala University for the idea).
//							Now negative EMG peaks can be fitted.
//							Fixed bug: 'use cursor' was not working for small delta-scaling.
// 2022-11-11 - ver. 1.06:	Fixed bug: EMG and ExpConvExp guesses did not work for XY data when the y-wave was scaled.
//							Added peak-area output to the EMG peak.
//							Added option to align the fit trace to the source trace (thanks to Oleg Dyachok from Uppsala University for the idea).
// 2022-11-27 - ver. 1.07:	Added support for fitting traces from 2D waves.
//							Added additional derived parameters for ExpConvExp (thanks to Oleg Dyachok for help).
//							Added confidence intervals to the area of the ExpModGauss and ExpConvExp functions.
//							Changed behavior: Now the cursor placement does not set the baseline, but only decides the fit range.
// 2022-12-07 - ver. 1.08:	Fixed bug: Derived parameters for ExpConvExp were wrong.
//							Now SuperQuickFit() can be invoked in user code and the command line.
//							Fixed bug: 2D data could not be properly read if the name was liberal.
// 2023-04-06 - ver. 1.10:	Added option to create a separate fit wave for each column of a 2D wave.
//							Traces are appended to the correct axis if the source is not plotted on the default axis.
//							New settings menu to choose color and style for the fit trace.
//							Added support for contour plots.
//							Global preferences are not overwritten completely by local references anymore.
//							Rearranged entries in the settings menu.
// 2023-11-03 - ver. 1.11:	Improved initial guesses for the DoubleGauss peak coefficients.
//							Added support for data plotted with column tags.
//							Added support for x-wave columns (traces plotted against a column of a 2D wave).
//							Fixed bug: It was not possible to set the start and end points with cursors directly.
//							Now the proper axis of the image / trace is used.
//							Column tags, instead of column numbers, are used for CurveFit commands if available.
// 2023-11-24 - ver. 1.12:	Fixed bug: Local settings could not be cleared.
//							Added panel for global settings.
//							Introduced pntCol_xw to the SQF structure to designate column of a 2D x-wave for fitting.
//							Fixed bug: custom fit functions did not fully support 2D x-waves.
//							Now a generic (fit-function agnostic) post-processing is executed, when a function with the name genericPostprocName + derivedVal_SUFFIX exists.
//							More robust handling of derivedVals functions.
//							Added option to enforce fit trace style.
//							Fitting individual columns now creates separate result annotations for each column.
//							Correct trace markers are displayed now inside annotations when 'Fit Individual 2D Columns' is active.
// 2024-03-05 - ver. 1.13:	Added support for 2D user functions.
// 2024-06-16 - ver. 1.14:	Fixed bug: The first 1D user function in the menu list was always deactivated.
// 2024-12-02 - ver. 1.20:	Added support for fitting plots with a limited y-range.
//							The main menu is now grayed out if a fit trace is selected (analogous to Quick Fit).
//							Now both cursors must be present for limiting the range (cursor-range mode).
//							Added support for limited error-wave ranges in the plot.
//							Rewrote and fixed several bugs in trace-range-finding code.
//							Now the axis-range mode supports trace offsets and scaling.
//							Added support for wave increments, i.e., skipping steps when plotting data.
//							Added support for traces from multidimensional waves (3D and 4D).
//							The structure members pntCol, pntCol_xw and pntCol_yw are removed from the SQF structure! use the new yw_dims and xw_dims instead.
//							Fixed limited range selection via cursors for contour plots.
//							Fixed bug: Error bars are now interpreted as 'standard deviation' similar to Quick Fit.
//							The fit command uses relative folder paths now (-> toggle this on/off via fitUseRelativePaths).
//							Fixed faulty cursor position interpretation for free cursors and cursors on different traces.
//							Added new trace style setting: Option to sort the fit trace below the input trace (to avoid clicking on the fit trace).
//							Further improved the peak finder for the DoubleGauss function.
//							Added the MultiGauss fit function.
//							Fixed bug: Contour traces we not properly registered if the wave name was too long.
//							Added support for XYZ contour traces.
//							Added warning message when trace offsets might interfere with the 'plot full range' setting.
// 2025-02-14 - ver. 1.21:	SQF menu is also greyed out if top trace is starting with 'fit_' and graph menu is invoked.
//							Fixed bug: W_fitConstants was not cleaned up.
// 2025-08-25 - ver. 1.24:	Fixed bug: Bad definition of color popups which were problematic in Igor 8.
//							Improved 'lower saturation' color mode for fit traces.
//							Code optimizations using the IPT tool (ver. 0.9) from BytePhysics. No errors or warnings for:
//								ipt lint --exclude CodeStyleDefaultPragmas '.\Super Quick Fit.ipf'
//							Added the fit wave to SuperQuickFitStruct.
//							Now ExpConvExp can be fitted to data with an inverted x axis.
// 2026-01-07 - ver. 1.25:	Code check using the IPT tool (ver. 0.10) from BytePhysics. No errors or warnings for: ipt lint '.\Super Quick Fit.ipf'
//							Now waves can be used as fit in the new Use_Wave_Data fit function.
//__________________________________________________________________________________________________

// *** global definitions **************************************************************************
StrConstant QuickFit_DefaultFuncs1D = "Gauss;Lor;Voigt;Sin;Line;-;HillEquation;Sigmoid;Power;LogNormal;Log;-;Exp;Exp_xOffset;DblExp;DblExp_Peak;DblExp_xOffset;"
StrConstant QuickFit_DefaultFuncs2D = "Gauss2D;"

static StrConstant CoefGen_SUFFIX      = "_prepareCoef"												// name suffixes for coef and output functions (see dummies below)
static StrConstant derivedVal_SUFFIX   = "_derivedVals"
static StrConstant genericPostprocName = "SuperQuickFit"											// use for generic (fit-function agnostic) post-processing
static StrConstant localSettingsPath   = "root:QuickFitMenu_settings"								// where local settings are saved
static StrConstant validCursorNameList = "A;B;C;D;E;F;G;H;I;J;"
static Constant fitUseRelativePaths = 1																// toggles use of relative paths in fit command

Function dummyFunc_prepareCoef(STRUCT SuperQuickFitStruct &s)										// dummy functions for coefficient preparation and output post-processing

	return -1	// error
End

Function/S dummyFunc_derivedVals(STRUCT SuperQuickFitStruct &s)

	return ""
End

Structure SuperQuickFitStruct
	string fitFunc			// used fit function name
	WAVE cw					// coef wave
	WAVE sw					// covariance matrix (M_Covar)
	WAVE data				// data wave (1D or 2D)
	WAVE fit				// fit wave (only if fit was successful!)
	WAVE xw					// x wave
	WAVE yw					// y wave (for images and contours; not supported for images yet)
	int32 pntMin			// fit range (x direction for images): start point
	int32 pntMax			// fit range (x direction for images): end point
	int32 pntMinY			// y direction for images: start point
	int32 pntMaxY			// y direction for images: end point
	int32 xw_Min			// fit range of trace's x wave (ver. 1.20)
	int32 xw_Max
	int32 yw_inc			// increment of the plotted traces (ver. 1.20)
	int32 xw_inc
	int16 yw_trDim			// used dimension for trace -> y-wave (ver. 1.20)
	int16 xw_trDim			// used dimension -> x-wave (if available)
	int32 yw_dims[4]		// y-wave dimensions for traces (multidimensional waves, ver. 1.20) => records the start point of every dimension
	int32 xw_dims[4]		// x-wave dimensions for traces
	double xBase			// x baseline / start offset
	double yBase			// y baseline / start offset
	int16 holdMode			// current hold setting: [0] = no hold, [1] = force o zero, [2] = force to min
	int16 plotFull			// setting to plot over full range
	int16 doTextbox			// setting for text-box drawing
	string holdStr			// hold string for the fit
EndStructure

// *** persistent preferences support *** ##########################################################

static StrConstant defaultSettings	= "offsetMode:0;plotFull:0;plotAligned:0;fitRangeMode:1;addTextBox:0;weightError:1;outCleanup:0;textboxOpt:791;multiColMode:0;trStyleMode:0;trLineSize:1;trColorMode:0;trColor:65535,0,0,65535;trForceStyle:0;trSortBehind:0;"
static StrConstant kPackageName		= "SuperQuickFit"
static StrConstant kPrefsFileName	= "SuperQuickFitPrefs.bin"
static Constant kPrefsVersion		= 100															// version 1.00

Structure SuperQuickFitPrefs
	uint32	version					// Preferences structure version number. 100 means 1.00.
	int16	offsetMode
	int16	plotFull
	int16	fitRangeMode
	int16	addTextBox
	int16	weightError
	int16	outCleanup
	int16	textboxOpt
	int16	plotAligned				// new since version 1.06
	int16	trStyleMode				// version 1.09
	int16	trColorMode				// version 1.09
	double	trLineSize				// version 1.09
	uint16	trColor[4]				// version 1.09
	int16	multiColMode			// version 1.09
	int16	trForceStyle			// version 1.12
	int16	trSortBehind			// version 1.20
	char reserved[200-4*1-17*2-8*1]	// Reserved for future use (int16 = 2, int32 = 4, double = 8)
EndStructure

static Function/S SQF_LoadPreferences()

	STRUCT SuperQuickFitPrefs s
	LoadPackagePreferences kPackageName, kPrefsFileName, 0, s
	string settings = ""
	if (V_flag!=0 || V_bytesRead==0 || s.version!=kPrefsVersion)									// If error or prefs not found or not valid, initialize them.
		SuperQuickFit#SQF_SavePreferences(defaultSettings)
	endif
	sPrintf settings, "offsetMode:%d;plotFull:%d;plotAligned:%d;fitRangeMode:%d;addTextBox:%d;weightError:%d;outCleanup:%d;textboxOpt:%d;multiColMode:%d;", s.offsetMode, s.plotFull, s.plotAligned, s.fitRangeMode, s.addTextBox, s.weightError, s.outCleanup, s.textboxOpt, s.multiColMode
	sPrintf settings, "%strStyleMode:%d;trLineSize:%d;trColorMode:%d;trColor:%d,%d,%d,%d;trForceStyle:%d;trSortBehind:%d;", settings, s.trStyleMode, s.trLineSize, s.trColorMode, s.trColor[0], s.trColor[1], s.trColor[2], s.trColor[3], s.trForceStyle, s.trSortBehind
	
	return settings
End

static Function SQF_SavePreferences(string settings)

	STRUCT SuperQuickFitPrefs s
	s.version		= kPrefsVersion
	s.offsetMode	= NumberByKey("offsetMode",		settings)
	s.plotFull		= NumberByKey("plotFull",		settings)
	s.fitRangeMode	= NumberByKey("fitRangeMode",	settings)
	s.addTextBox	= NumberByKey("addTextBox",		settings)
	s.weightError	= NumberByKey("weightError",	settings)
	s.outCleanup	= NumberByKey("outCleanup",		settings)
	s.textboxOpt	= NumberByKey("textboxOpt",		settings)
	s.plotAligned	= NumberByKey("plotAligned",	settings)
	s.trForceStyle	= NumberByKey("trForceStyle",	settings)
	s.trSortBehind	= NumberByKey("trSortBehind",	settings)
	s.trStyleMode	= NumberByKey("trStyleMode",	settings)
	s.trLineSize	= NumberByKey("trLineSize",		settings)
	s.trColorMode	= NumberByKey("trColorMode",	settings)
	s.multiColMode	= NumberByKey("multiColMode",	settings)
	string colors	= StringByKey("trColor",		settings)
	s.trColor[0]	= str2num(StringFromList(0, colors, ","))
	s.trColor[1]	= str2num(StringFromList(1, colors, ","))
	s.trColor[2]	= str2num(StringFromList(2, colors, ","))
	s.trColor[3]	= str2num(StringFromList(3, colors, ","))
	SavePackagePreferences kPackageName, kPrefsFileName, 0, s
	
	return V_flag
End

// *** SQF menu definitions *** ####################################################################

Menu "GraphPopup", dynamic
	SubMenu SuperQuickFit#QuickFitMenuSubMenuEntries("Super Quick Fit")
		SubMenu "Settings"
			SuperQuickFit#QuickFitMenuItem("useCursor"),	/Q, SuperQuickFit#QuickFitMenuSettings("useCursor")
			SuperQuickFit#QuickFitMenuItem("useAxRange"),	/Q, SuperQuickFit#QuickFitMenuSettings("useAxRange")
			SuperQuickFit#QuickFitMenuItem("multiColMode"),	/Q, SuperQuickFit#QuickFitMenuSettings("multiColMode")
			"Fit Trace Preferences ...",					/Q, SuperQuickFit#QickFitOutputTraceOptions()
			"-"
			SuperQuickFit#QuickFitMenuItem("zeroBase"),		/Q, SuperQuickFit#QuickFitMenuSettings("zeroBase")
			SuperQuickFit#QuickFitMenuItem("fixedBase"),	/Q, SuperQuickFit#QuickFitMenuSettings("fixedBase")
			SuperQuickFit#QuickFitMenuItem("weightError"),	/Q, SuperQuickFit#QuickFitMenuSettings("weightError")
			"-"
			SuperQuickFit#QuickFitMenuItem("outCleanup"),	/Q, SuperQuickFit#QuickFitMenuSettings("outCleanup")
			SuperQuickFit#QuickFitMenuItem("plotFull"),		/Q, SuperQuickFit#QuickFitMenuSettings("plotFull")
			SuperQuickFit#QuickFitMenuItem("plotAligned"),	/Q, SuperQuickFit#QuickFitMenuSettings("plotAligned")
			SuperQuickFit#QuickFitMenuItem("addTextBox"),	/Q, SuperQuickFit#QuickFitMenuSettings("addTextBox")
			"\t» Textbox Preferences ...",					/Q, SuperQuickFit#QickFitTextBoxOptions()
			"-"
			"Clear Local Settings",							/Q, SuperQuickFit#ClearLocalSettings()
		End
		"Settings Panel ...",								/Q, SuperQuickFit#QickFitGlobalOptions()
		"-"
		SuperQuickFit#QuickFitMenuUserEntries(), /Q, SuperQuickFit#SuperQuickFit_Wrapper("")
		"-"
		SubMenu SuperQuickFit#QuickFitMenuSubMenuEntries("Poly")
			"1;2;3;4;5;6;7;8;9;10;", /Q, SuperQuickFit#SuperQuickFit_Wrapper("poly")
		End
		SubMenu SuperQuickFit#QuickFitMenuSubMenuEntries("Poly_xOffset")
			"1;2;3;4;5;6;7;8;9;10;", /Q, SuperQuickFit#SuperQuickFit_Wrapper("poly_XOffset")
		End
		SubMenu SuperQuickFit#QuickFitMenuSubMenuEntries("Poly2D")
			"1;2;3;4;5;6;7;8;9;10;", /Q, SuperQuickFit#SuperQuickFit_Wrapper("Poly2D")
		End
		SubMenu SuperQuickFit#QuickFitMenuSubMenuEntries("MultiGauss")
			"1;2;3;4;5;6;7;8;9;10;", /Q, SuperQuickFit#SuperQuickFit_Wrapper("MultiGauss")
		End
		"-"
		SuperQuickFit#QuickFitMenuDefaultEntries(), /Q, SuperQuickFit#SuperQuickFit_Wrapper("")
	End
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

Menu "TracePopup", dynamic
	SubMenu SuperQuickFit#QuickFitMenuSubMenuEntries("Super Quick Fit")
		SubMenu "Settings"
			SuperQuickFit#QuickFitMenuItem("useCursor"),	/Q, SuperQuickFit#QuickFitMenuSettings("useCursor")
			SuperQuickFit#QuickFitMenuItem("useAxRange"),	/Q, SuperQuickFit#QuickFitMenuSettings("useAxRange")
			SuperQuickFit#QuickFitMenuItem("multiColMode"),	/Q, SuperQuickFit#QuickFitMenuSettings("multiColMode")
			"Fit Trace Preferences ...",					/Q, SuperQuickFit#QickFitOutputTraceOptions()
			"-"
			SuperQuickFit#QuickFitMenuItem("zeroBase"),		/Q, SuperQuickFit#QuickFitMenuSettings("zeroBase")
			SuperQuickFit#QuickFitMenuItem("fixedBase"),	/Q, SuperQuickFit#QuickFitMenuSettings("fixedBase")
			SuperQuickFit#QuickFitMenuItem("weightError"),	/Q, SuperQuickFit#QuickFitMenuSettings("weightError")
			"-"
			SuperQuickFit#QuickFitMenuItem("outCleanup"),	/Q, SuperQuickFit#QuickFitMenuSettings("outCleanup")
			SuperQuickFit#QuickFitMenuItem("plotFull"),		/Q, SuperQuickFit#QuickFitMenuSettings("plotFull")
			SuperQuickFit#QuickFitMenuItem("plotAligned"),	/Q, SuperQuickFit#QuickFitMenuSettings("plotAligned")
			SuperQuickFit#QuickFitMenuItem("addTextBox"),	/Q, SuperQuickFit#QuickFitMenuSettings("addTextBox")
			"\t» Textbox Preferences ...",					/Q, SuperQuickFit#QickFitTextBoxOptions()
			"-"
			"Clear Local Settings",							/Q, SuperQuickFit#ClearLocalSettings()
		End
		"Settings Panel ...",								/Q, SuperQuickFit#QickFitGlobalOptions()
		"-"
		SuperQuickFit#QuickFitMenuUserEntries(), /Q, SuperQuickFit#SuperQuickFit_Wrapper("")
		"-"
		SubMenu SuperQuickFit#QuickFitMenuSubMenuEntries("Poly")
			"1;2;3;4;5;6;7;8;9;10;", /Q, SuperQuickFit#SuperQuickFit_Wrapper("poly")
		End
		SubMenu SuperQuickFit#QuickFitMenuSubMenuEntries("Poly_xOffset")
			"1;2;3;4;5;6;7;8;9;10;", /Q, SuperQuickFit#SuperQuickFit_Wrapper("poly_XOffset")
		End
		SubMenu SuperQuickFit#QuickFitMenuSubMenuEntries("Poly2D")
			"1;2;3;4;5;6;7;8;9;10;", /Q, SuperQuickFit#SuperQuickFit_Wrapper("Poly2D")
		End
		SubMenu SuperQuickFit#QuickFitMenuSubMenuEntries("MultiGauss")
			"1;2;3;4;5;6;7;8;9;10;", /Q, SuperQuickFit#SuperQuickFit_Wrapper("MultiGauss")
		End
		"-"
		SuperQuickFit#QuickFitMenuDefaultEntries(), /Q, SuperQuickFit#SuperQuickFit_Wrapper("")
	End
End

// *** SQF menu interaction *** ####################################################################

static Function ClearLocalSettings()

	KillStrings/Z $localSettingsPath
End

static Function SuperQuickFit_Wrapper(string option)

	GetLastUserMenuInfo
	int isMultFunc = (StringMatch(option, "poly*") || StringMatch(option, "multi*")) && !numtype(str2num(S_value))
	string fitFunc = SelectString(isMultFunc, "", option + " ") + S_value							// grab fitFunc from selected item: for poly and multi fits append degree n
	
	SuperQuickFit(S_graphName, S_traceName, fitFunc)
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

static Function/S QuickFitMenuSubMenuEntries(string entry)											// sub menu entry generator

	int disable, noTraces, notFor2D;	[noTraces, notFor2D] = SuperQuickFit#CheckTraceOrImage()
	if (!CmpStr(entry, "Super Quick Fit"))															// SQF main menu
		GetLastUserMenuInfo
		string top = SelectString(strlen(S_traceName), StringFromList(0, TraceNameList(S_graphName,";",1)), S_traceName)
		disable = (noTraces && notFor2D==1) || StringMatch(top, "fit_*")							// nothing usable
	else
		disable = ((StringMatch(entry, "*2D") && !notFor2D) %^ noTraces)							// either both no traces and 2D-compatible (1 %^ 1) or both traces and 2D-incompatible (0 %^ 0)
	endif
	
	return SelectString(disable,"","(") + entry
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

static Function/S QuickFitMenuUserEntries()															// entries from user functions

	string funcList = RemoveFromList("dummyFunc;MultiGauss;", ReplaceString(CoefGen_SUFFIX, FunctionList("*" + CoefGen_SUFFIX, ";", ""), ""))
	string funcs_1D = ""
	string funcs_2D = ""
	int i, noTraces, notFor2D
	for (i = 0; i < ItemsInList(funcList); i++)
		string func = StringFromList(i,funcList)
		int type = SQF_GetUserFuncType(func)
		if (type == 1)
			funcs_1D += func + ";"
		endif
		if (type == 2)
			funcs_2D += func + ";"
		endif
	endfor
	[noTraces, notFor2D] = SuperQuickFit#CheckTraceOrImage()
	
	return SuperQuickFit#deactivateList(funcs_2D, !noTraces || notFor2D) + SuperQuickFit#deactivateList(funcs_1D, noTraces)
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

static Function/S QuickFitMenuDefaultEntries()														// default fit list

	int noTraces, notFor2D; [noTraces, notFor2D] = SuperQuickFit#CheckTraceOrImage()
	return SuperQuickFit#deactivateList(QuickFit_DefaultFuncs2D, !noTraces || notFor2D) + SuperQuickFit#deactivateList(QuickFit_DefaultFuncs1D, noTraces)
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

static Function/S SQF_getTopGraph()

	return StringFromList(0,WinList("*", ";", "WIN:1"))
End

static Function [int noTraces, int notFor2D] CheckTraceOrImage()									// checks for compatible image or trace in top graph

	string gName = SQF_getTopGraph()
	if (!strlen(gName))
		return [1, 1]
	endif
	string image = ImageInfo(gName,"",0)
	noTraces = ItemsInList(TraceNameList(gName,";",1))==0
	notFor2D = 0
	if (noTraces)
		if (strlen(image))
			WAVE/Z imgZW = $(StringByKey("ZWAVEDF",image) + PossiblyQuoteName(StringByKey("ZWAVE",image)))
			WAVE/Z imgXW = $(StringByKey("XWAVEDF",image) + PossiblyQuoteName(StringByKey("XWAVE",image)))
			WAVE/Z imgYW = $(StringByKey("YWAVEDF",image) + PossiblyQuoteName(StringByKey("YWAVE",image)))
			notFor2D = WaveExists(imgXW) || WaveExists(imgYW)										// no support for x/y waves with images yet
			if (WaveExists(imgZW))
				notFor2D = (WaveDims(imgZW) > 2) ? 1 : notFor2D										// higher dimension images not supported
			endif
			notFor2D *= 2																			// >1 doesn't disable the main menu
		else
			notFor2D = !ItemsInList(ContourNameList(gName,";"))										// otherwise look for contour plots
		endif
	endif
	
	return [noTraces, notFor2D]
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

static Function/S deactivateList(string list, int deactivate)										// deactivate menu entry list (prepend '(' character)

	if (deactivate)
		return SelectString(ItemsInList(list), "", "(") + RemoveEnding(ReplaceString(";", list, ";("), "(")
	endif
	return list
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

static Function/S QuickFitMenuItem(string which)													// returns menu entry for settings

	string set = StrVarOrDefault(localSettingsPath, SuperQuickFit#SQF_LoadPreferences())
	strswitch(which)
		case "zeroBase":
			return SelectString((NumberByKey("offsetMode"	,set)==1),"","!*") + "Offset: Force to Zero"
		case "fixedBase":
			return SelectString((NumberByKey("offsetMode"	,set)==2),"","!*") + "Offset: Force to Minimum"
		case "plotFull":
			return SelectString((NumberByKey("plotFull"		,set)==1),"","!*") + "Result: Plot Full Range"
		case "plotAligned":
			return SelectString((NumberByKey("plotAligned"	,set)==1),"","!*") + "Result: Align Fit with Trace"
		case "addTextBox":
			return SelectString((NumberByKey("addTextBox"	,set)==1),"","!*") + "Result: Add TextBox"
		case "multiColMode":
			return SelectString((NumberByKey("multiColMode"	,set)==1),"","!*") + "Fit Individual 2D Columns"
		case "useCursor":
			return SelectString((NumberByKey("fitRangeMode"	,set)==1),"","!*") + "Fit Between Cursors"
		case "useAxRange":
			return SelectString((NumberByKey("fitRangeMode"	,set)==2),"","!*") + "Fit Within Axis Range"
		case "weightError":
			return SelectString((NumberByKey("weightError"	,set)==1),"","!*") + "Weight from Error Bar Wave"
		case "outCleanup":
			return SelectString((NumberByKey("outCleanup"	,set)==1),"","!*") + "Clean up Output Data"
		default:
			return ""
	endswitch
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

static Function QuickFitMenuSettings(string which)													// write settings from selection

	string globalVal = SuperQuickFit#SQF_LoadPreferences()											// saves global (PackagePrefs) and local (string) settings separately
	string localVals = StrVarOrDefault(localSettingsPath, globalVal)
	localVals = SuperQuickFit#SQF_doGeneralSettings(which,localVals)
	globalVal = SuperQuickFit#SQF_doGeneralSettings(which,globalVal)
	string/G $localSettingsPath = localVals
	return SuperQuickFit#SQF_SavePreferences(globalVal)
End

static Function/S SQF_doGeneralSettings(string which, string set)									// replace selected setting parameter

	strswitch(which)
		case "zeroBase":
			return ReplaceNumberByKey("offsetMode"	,set ,((NumberByKey("offsetMode"	,set)==1) ? 0 : 1))
		case "fixedBase":
			return ReplaceNumberByKey("offsetMode"	,set ,((NumberByKey("offsetMode"	,set)==2) ? 0 : 2))
		case "useCursor":
			return ReplaceNumberByKey("fitRangeMode",set ,((NumberByKey("fitRangeMode"	,set)==1) ? 0 : 1))
		case "useAxRange":
			return ReplaceNumberByKey("fitRangeMode",set ,((NumberByKey("fitRangeMode"	,set)==2) ? 0 : 2))
		default:
			return ReplaceNumberByKey(which			,set ,((NumberByKey(which 			,set)==1) ? 0 : 1))
	endswitch
End

// *** menu: text options panel *** ################################################################

static StrConstant kOptBits = "Title;Date;Time;FitType;FitFunc;ModelWave;YWave;XWave;CoefValues;CoefErrors;"

static Function QickFitTextBoxOptions()

	if (WinType("SQF_TextBoxOpt"))
		return 0
	endif
	
	string title = "Select All;Title;Date and Time;Date;Time;Fit Type;Fit Function Name;Waves;Model Wave;Y Wave;X Wave(s);Coefficient Report;Coefficient Values;Coefficient Errors;"
	string names = "All;Title;DateTime;Date;Time;FitType;FitFunc;Waves;ModelWave;YWave;XWave;CoefReport;CoefValues;CoefErrors;"
	string align = "0;0;0;15;15;0;0;0;15;15;15;0;15;15;"
	GetMouse
	NewPanel/FLT=2/K=1/W=(V_left,V_top,V_left + 155,V_top + 400)/N=SQF_TextBoxOpt as "Textbox Preferences"
	int i; int d = 25
	for (i = 0; i < ItemsInList(title); i++)
		variable cP = 15 + str2num(StringFromList(i,align))
		CheckBox $("c"+StringFromList(i,names)) ,pos={cP,15+d*i} ,size={100,23} ,title=StringFromList(i,title) ,proc=SuperQuickFit#SQF_TextboxOptions
	endfor
	Button b_Save	,pos={10,15+d*i} ,size={60,22} ,title="Save"	,proc=SuperQuickFit#SQF_TextBoxButton
	Button b_Cancel	,pos={85,15+d*i} ,size={60,22} ,title="Cancel"	,proc=SuperQuickFit#SQF_TextBoxButton
	SuperQuickFit#SQF_TextBoxLoadFromSettings()
	
	return 0
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

static Function SQF_TextboxOptions(STRUCT WMCheckboxAction &s) : CheckBoxControl

	if(s.eventCode != 2)
		return 0
	endif
	
	string ctrlList
	strswitch(s.ctrlName)
		case "cDateTime":
			ctrlList = "cDate;cTime;"
		break
		case "cWaves":
			ctrlList = ControlNameList(s.win, ";", "*Wave")
		break
		case "cCoefReport":
			ctrlList = ControlNameList(s.win, ";", "cCoef*")
		break
		case "cAll":
			ctrlList = ControlNameList(s.win, ";", "c*")
		break
		default:
			ctrlList = ""
		break
	endswitch
	
	int i; int count = 0
	for (i=0; i<ItemsInList(ctrlList); i++)
		CheckBox $(StringFromList(i,ctrlList)) ,win=$s.win ,value=s.checked
	endfor
	ctrlList = RemoveFromList("cAll", ControlNameList(s.win, ";", "c*"))							// now see if the Check All box needs to be toggled
	for (i=0; i<ItemsInList(ctrlList); i++)
		ControlInfo/W=$s.win $("c" + StringFromList(i,kOptBits)); count += V_value
	endfor
	CheckBox cAll ,win=$s.win ,value=(ItemsInList(ctrlList) == count)
	
	return 0
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

static Function SQF_TextBoxButton(STRUCT WMButtonAction &s) : ButtonControl

	if (s.eventCode == 2)
		if(!CmpStr(s.ctrlName,"b_Save"))
			SuperQuickFit#SQF_TextBoxWriteSettings()												// save options
		endif
		KillWindow $s.win
	endif
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

static Function SQF_TextBoxWriteSettings()															// save textbox bit settings

	variable exponent
	variable textOpts = 0
	for (exponent = 0; exponent < ItemsInList(kOptBits); exponent++)								// read checked bits from checkboxes
		ControlInfo/W=SQF_TextBoxOpt $("c" + StringFromList(exponent, kOptBits))
		textOpts = V_Value ? (textOpts | (2^exponent)) : textOpts
	endfor
	
	string globalVal = SuperQuickFit#SQF_LoadPreferences()
	string localVals = StrVarOrDefault(localSettingsPath, globalVal)
	localVals = ReplaceNumberByKey("textboxOpt", localVals, textOpts)
	globalVal = ReplaceNumberByKey("textboxOpt", globalVal, textOpts)
	string/G  $localSettingsPath = localVals														// write new option
	return SuperQuickFit#SQF_SavePreferences(globalVal)
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

static Function SQF_TextBoxLoadFromSettings()														// transfers checked bits to checkboxes

	string   settings = StrVarOrDefault(localSettingsPath,SuperQuickFit#SQF_LoadPreferences())
	variable exponent
	variable textOpts = NumberByKey("textboxOpt",settings)
	for (exponent = 0; exponent < ItemsInList(kOptBits); exponent++)
		CheckBox $("c" + StringFromList(exponent, kOptBits)) ,win=SQF_TextBoxOpt ,value=((textOpts & 2^exponent) != 0)
	endfor
	CheckBox cDateTime	 ,win=SQF_TextBoxOpt ,value=((textOpts & 2^1)!=0 && (textOpts & 2^2)!=0)
	CheckBox cWaves		 ,win=SQF_TextBoxOpt ,value=((textOpts & 2^5)!=0 && (textOpts & 2^6)!=0 && (textOpts & 2^7)!=0)
	CheckBox cCoefReport ,win=SQF_TextBoxOpt ,value=((textOpts & 2^8)!=0 && (textOpts & 2^9)!=0)
	CheckBox cAll		 ,win=SQF_TextBoxOpt ,value=( textOpts == 1023)								// all 10 bits are set
End

// *** menu: trace options panel *** ###############################################################

static StrConstant kTraceColorOpt = "Default Color;Single Fixed Color;Same Color as Input;Complementary to Input;Lower Saturation than Input;"

static Function QickFitOutputTraceOptions()

	if (WinType("SQF_FitTraceOpt"))
		return 0
	endif
	
	GetMouse
	NewPanel/FLT=2/K=1/W=(V_left,V_top,V_left + 300,V_top + 150)/N=SQF_FitTraceOpt as "Fit Trace Preferences"
	int i = 0
	int d = 26
	PopupMenu p_TrColor		,pos={235,10+d*i}	,size={45,23}	,title=""				,bodyWidth=50	,value=#"\"*COLORPOP*\""
	PopupMenu p_TrColorMode	,pos={15, 10+d*i++}	,size={265,23}	,title="Color Mode:"	,bodyWidth=195	,value=#("\"" + kTraceColorOpt + "\"")
	PopupMenu p_TrStyle		,pos={15, 10+d*i}	,size={149,23}	,title="Line Style:"	,bodyWidth=80	,value=#"\"*LINESTYLEPOP*\""
	SetVariable v_TrLSize	,pos={175,10+d*i++}	,size={105,23}	,title="Line Size:"		,bodyWidth=50	,value=_NUM:1	,limits={0,10,1}
	CheckBox c_TrForceStyle	,pos={83, 10+d*i++}	,size={265,23}	,title="Enforce Style Each Time"			,help={"Updates the style of the fit trace even if the trace is already plotted."}
	CheckBox c_TrSortBehind	,pos={83, 10+d*i++}	,size={265,23}	,title="Move Fit Trace Behind Input Trace"	,help={"Sorts the traces so that the fit trace will be behind the input trace in the graph."}
	Button b_Save			,pos={50, 12+d*i} 	,size={80,22}	,title="Save"
	Button b_Cancel			,pos={170,12+d*i}	,size={80,22}	,title="Cancel"
	ModifyControlList ControlNameList("SQF_FitTraceOpt",";","b_*") proc=SuperQuickFit#SQF_FitOptionsButton
	ModifyControlList ControlNameList("SQF_FitTraceOpt",";","p_*") proc=SuperQuickFit#SQF_FitOptionsPopup
	
	SuperQuickFit#SQF_FitTraceLoadFromSettings()
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

static Function SQF_FitOptionsButton(STRUCT WMButtonAction &s) : ButtonControl

	if (s.eventCode == 2)
		if(!CmpStr(s.ctrlName,"b_Save"))
			SuperQuickFit#SQF_FitTraceWriteSettings()												// save options
		endif
		KillWindow $s.win
	endif
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

Function SQF_FitOptionsPopup(STRUCT WMPopupAction &s) : PopupMenuControl

	if (s.eventCode == 2 && !CmpStr(s.ctrlName,"p_TrColorMode"))
		int off = !StringMatch(s.popStr,"Single Fixed Color")
		PopupMenu p_TrColor 	,win=$s.win ,disable=off
		PopupMenu p_TrColorMode ,win=$s.win ,pos={15,10} ,size={205+60*off,23} ,bodyWidth=135+60*off
	endif
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

static Function SQF_FitTraceWriteSettings()

	string globalVal = SuperQuickFit#SQF_LoadPreferences()
	string localVals  = StrVarOrDefault(localSettingsPath,globalVal)
	localVals = SuperQuickFit#SQF_doTraceStyleSettings(localVals)
	globalVal = SuperQuickFit#SQF_doTraceStyleSettings(globalVal)
	string/G $localSettingsPath  = localVals
	return SuperQuickFit#SQF_SavePreferences(globalVal)
End

static Function/S SQF_doTraceStyleSettings(string settings)											// replace fit trace settings

	string color = ""
	ControlInfo/W=SQF_FitTraceOpt c_TrSortBehind;	settings = ReplaceNumberByKey("trSortBehind",settings,V_Value)
	ControlInfo/W=SQF_FitTraceOpt c_TrForceStyle;	settings = ReplaceNumberByKey("trForceStyle",settings,V_Value)
	ControlInfo/W=SQF_FitTraceOpt v_TrLSize;		settings = ReplaceNumberByKey("trLineSize",settings,V_Value)
	ControlInfo/W=SQF_FitTraceOpt p_TrStyle;		settings = ReplaceNumberByKey("trStyleMode",settings,V_Value-1)
	ControlInfo/W=SQF_FitTraceOpt p_TrColorMode;	settings = ReplaceNumberByKey("trColorMode",settings,V_Value-1)
	ControlInfo/W=SQF_FitTraceOpt p_TrColor;		sPrintf color, "%d,%d,%d,%d", V_Red, V_Green, V_Blue, V_Alpha
	
	return ReplaceStringByKey("trColor",settings,color)
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

static Function SQF_FitTraceLoadFromSettings()

	string settings = StrVarOrDefault(localSettingsPath,SuperQuickFit#SQF_LoadPreferences())
	int ForceSty = NumberByKey("trForceStyle",settings);	ForceSty = (ForceSty <  0) ? 0 : ForceSty
	int FitBelow = NumberByKey("trSortBehind",settings);	FitBelow = (FitBelow <  0) ? 0 : FitBelow
	int FitColor = NumberByKey("trColorMode",settings);		FitColor = (FitColor <  0) ? 0 : FitColor
	int FitStyle = NumberByKey("trStyleMode",settings);		FitStyle = (FitStyle <  0) ? 0 : FitStyle
	int FitLSize = NumberByKey("trLineSize",settings);		FitLSize = (FitLSize <= 0) ? 1 : limit(FitLSize,0,10)
	string color = StringByKey("trColor",settings)
	int A = str2num(StringFromList(2,color,","))
	int R = str2num(StringFromList(0,color,",")); R = (R < 0 || A <= 0) ? 65535 : R
	int G = str2num(StringFromList(1,color,",")); G = (G < 0 || A <= 0) ? 0 : G
	int B = str2num(StringFromList(2,color,",")); B = (B < 0 || A <= 0) ? 0 : B
	A = (A <= 0) ? 65535 : A
	int expand = 60*(FitColor!=1)
	PopupMenu p_TrColorMode	,win=SQF_FitTraceOpt	,mode=FitColor+1	,size={205+expand,23}	,bodyWidth=135+expand
	PopupMenu p_TrStyle		,win=SQF_FitTraceOpt	,mode=FitStyle+1
	PopupMenu p_TrColor		,win=SQF_FitTraceOpt	,popColor=(R,G,B,A)	,disable=(FitColor!=1)
	SetVariable v_TrLSize	,win=SQF_FitTraceOpt	,value=_NUM:FitLSize
	CheckBox c_TrForceStyle	,win=SQF_FitTraceOpt	,value=ForceSty
	CheckBox c_TrSortBehind	,win=SQF_FitTraceOpt	,value=FitBelow
End

// *** menu: global options panel *** ##############################################################

static Function QickFitGlobalOptions()

	if (WinType("SQF_GlobalOpt"))
		DoWindow/F SQF_GlobalOpt
		return 0
	endif
	
	string title = "Fit Full Range;Fit Between Cursors;Fit Within Axis Range;Fit Individual 2D Columns;Offset: Fit to Data;Offset: Force to Zero;Offset: Force to Minimum;Weight from Error Bar Wave;Clean up Output Data;Plot Full Range;Align Fit with Trace;Add TextBox;"
	string names = "fitRangeMode_0;fitRangeMode_1;fitRangeMode_2;multiColMode_1;offsetMode_0;offsetMode_1;offsetMode_2;weightError_1;outCleanup_1;plotFull_1;plotAligned_1;addTextBox_1;"
	string style = "1;1;1;0;1;1;1;0;0;0;0;0;0;0;0;0;"												// checkbox: radio button or check mark
	string group = "group_0:Fit Range,4;group_4:Fit Options,4;group_8:Results,6.5;"					// group names and heights (in distance units)
	
	variable pH = 544
	variable pV = 200
	int i
	int d = 25
	int o = 0
	
	GetMouse
	NewPanel/K=((IgorVersion() < 10.0) ? 1 : 4)/W=(V_left - pV/2, V_top - pH/2, V_left + pV/2, V_top + pH/2)/N=SQF_GlobalOpt as "SQF: Settings"
	for (i = 0; i < ItemsInList(title); i++)
		string curr = ReplaceString(",", StringByKey("group_" + num2str(i), group), ";")
		if (strlen(curr))
			variable cH = d*(str2num(StringFromList(1,curr)) + 1.1)
			o += 10
			GroupBox $("group_" + num2str(i))		,pos={8 ,o+d*i} ,size={183,cH} ,title=StringFromList(0,curr)  ,frame=0
			o += d
		endif
		CheckBox $("c_" + StringFromList(i,names))	,pos={15,o+d*i} ,size={170,23} ,title=StringFromList(i,title) ,mode=str2num(StringFromList(i,style)) ,proc=SuperQuickFit#SQF_GlobalOptions
	endfor
	o += d*i;	i = 0;	d += 8																		// adjust distances for the buttons
	Button b_TextBoxPrefs	,pos={18,o+d*i++} ,size={162,22} ,title="Textbox Preferences"	,proc=SuperQuickFit#SQF_GlobalSettingsButton
	Button b_FitTracePrefs	,pos={18,o+d*i++} ,size={162,22} ,title="Fit Trace Preferences"	,proc=SuperQuickFit#SQF_GlobalSettingsButton
	o += 8
	Button b_ClearLocPrefs	,pos={18,o+d*i++} ,size={162,22} ,title="Clear Local Settings"	,proc=SuperQuickFit#SQF_GlobalSettingsButton
	Button b_ResetAllPrefs	,pos={18,o+d*i++} ,size={162,22} ,title="Load Default Settings"	,proc=SuperQuickFit#SQF_GlobalSettingsButton
	
	SetWindow SQF_GlobalOpt hook(UpdateFunc)=SuperQuickFit#SQF_PanelEvents
	SuperQuickFit#SQF_LoadPanelSettings()
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

static Function SQF_PanelEvents(STRUCT WMWinHookStruct &s)

	if (s.EventCode==0)
		SuperQuickFit#SQF_LoadPanelSettings()
	endif
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

static Function BeforeExperimentSaveHook(variable rN, string fileName, string path, string type, string creator, variable kind)

	if (WinType("SQF_GlobalOpt") == 7)
		KillWindow/Z SQF_GlobalOpt																	// panel will be killed before save operation
		KillWindow/Z SQF_FitTraceOpt
		KillWindow/Z SQF_TextBoxOpt
	endif
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

static Function SQF_LoadPanelSettings()

	string settings = StrVarOrDefault(localSettingsPath,SuperQuickFit#SQF_LoadPreferences())
	string chcklist = ControlNameList("SQF_GlobalOpt", ";", "c_*"); int i
	for (i=0; i<ItemsInList(chcklist); i++)
		CheckBox $(StringFromList(i,chcklist)) ,win=SQF_GlobalOpt ,value=0							// first reset everything
	endfor
	for (i=0; i<ItemsInList(settings); i++)
		string para = StringFromList(0,StringFromList(i,settings),":")
		string ctrl = "c_" + para + "_" + StringByKey(para,settings)								// all checkbox names are of the format 'c_PARANAME_VAL'
		ControlInfo/W=SQF_GlobalOpt $ctrl
		if (V_flag==2)
			CheckBox $ctrl ,win=SQF_GlobalOpt ,value=1												// found checkbox for the right setting? => check
		endif
	endfor
	Button b_ClearLocPrefs ,win=SQF_GlobalOpt ,disable=2*(Exists(localSettingsPath)!=2)				// disable button if no local settings were found
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

static Function SQF_GlobalOptions(STRUCT WMCheckboxAction &s) : CheckBoxControl

	if(s.eventCode != 2)
		return 0
	endif
	string paraName = StringFromList(1,s.ctrlName,"_")												// all checkbox names are of the format 'c_PARANAME_VAL'
	int val = str2num(StringFromList(2,s.ctrlName,"_"))
	string globalVal = SuperQuickFit#SQF_LoadPreferences()											// saves global (PackagePrefs) and local (string) settings separately
	string localVals = StrVarOrDefault(localSettingsPath, globalVal)
	localVals = ReplaceNumberByKey(paraName ,localVals ,((NumberByKey(paraName ,localVals) == val) ? 0 : val))
	globalVal = ReplaceNumberByKey(paraName ,globalVal ,((NumberByKey(paraName ,globalVal) == val) ? 0 : val))
	string/G $localSettingsPath = localVals
	SuperQuickFit#SQF_SavePreferences(globalVal)
	SuperQuickFit#SQF_LoadPanelSettings()
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

static Function SQF_GlobalSettingsButton(STRUCT WMButtonAction &s) : ButtonControl

	if (s.eventCode != 2)
		return 0
	endif
	
	strswitch(s.ctrlName)
		case "b_TextBoxPrefs":
			SuperQuickFit#QickFitTextBoxOptions()
		break
		case "b_FitTracePrefs":
			SuperQuickFit#QickFitOutputTraceOptions()
		break
		case "b_ResetAllPrefs":																		// fallthrough, also clear settings
			SuperQuickFit#SQF_SavePreferences(defaultSettings)
		case "b_ClearLocPrefs":
			SuperQuickFit#ClearLocalSettings()
		break
		default:
		break
	endswitch
	
	SuperQuickFit#SQF_LoadPanelSettings()
End

//__________________________________________________________________________________________________
// *** SQF kernel (both for UI and user code) ***
//__________________________________________________________________________________________________

Function SuperQuickFit(string gName, string trace, string fitFunc)

	STRUCT SuperQuickFitStruct s
	string trcName, trcID;	SplitString/E="([^#]*)(.*)" trace, trcName, trcID
	trace = PossiblyQuoteName(trcName) + trcID														// make sure the trace name has the correct quotes
	
	// +++++++++++++++++++++++++++++ first some error handling +++++++++++++++++++++++++++++++++++++
	if (!strlen(gName))
		gName = SQF_getTopGraph()																	// try to get a valid graph
		if (!strlen(gName))
			Print "SQF error: Graph not found."
			return -2
		endif
	endif
	
	if (WinType(gName) != 1)
		Print "SQF error: Graph '" + gName + "' does not exist."
		return -3
	endif
	
	string traceList   = TraceNameList(gName,";",1)
	string contourList = TraceNameList(gName,";",2)
	if (strlen(trace) && FindListItem(trace, traceList + contourList,";",0,0) < 0)
		Print "SQF error: Trace " + trace + " not in graph " + gName + "."
		return -4
	endif
	
	int isDefaultFunc = (FindListItem(fitFunc, RemoveFromList("-", QuickFit_DefaultFuncs1D + QuickFit_DefaultFuncs2D), ";", 0, 0) > -1)
	int isPolyFitFunc = (GrepString(fitFunc,"(?i)poly[^ ]* [0-9]+"))
	int isMultiPeakFit= (GrepString(fitFunc,"(?i)multi[^ ]* [0-9]+"))
	int isUserFitFunc = (!isDefaultFunc && !isPolyFitFunc && !isMultiPeakFit && FindListItem(fitFunc, FunctionList("*",";","KIND:2"),";",0,0) > -1)
	
	string fitType = "CurveFit"
	int userFuncDim = isUserFitFunc ? SQF_GetUserFuncType(fitFunc) : 0
	if ((!isDefaultFunc && !isPolyFitFunc && !isUserFitFunc && !isMultiPeakFit) || (isUserFitFunc && !userFuncDim))
		Print "SQF error: Not a valid function type."
		return -5																					// not compatible
	endif
	
	if (isUserFitFunc || isMultiPeakFit)
		fitType = SelectString(userFuncDim==2,"FuncFit","FuncFitMD")
	endif
	
	string image = ""
	string contour = ""
	if (FindListItem(trace, contourList,";",0,0) > -1)												// contour plot
		SplitString/E="([^\=]*)(?:[^#]*)#?([0-9]*)?" trace, trcName, trcID							// split at = and possibly #
		int instance = strlen(trcID) ? str2num(trcID) : 0											// is instanced if two waves with a similar long name exist
		trcName = StringFromList(instance, GrepList(ContourNameList(gName,";"), "^" + trcName))		// find associated contour wave name
		contour = ContourInfo(gName,trcName,0)
		if (!strlen(contour) || !strlen(trcName))
			Print "SQF error: Not a valid contour trace."
			return -6
		endif
	endif
	
	if (!strlen(trace))																				// was called from graph popup => find top trace
		image = ImageInfo(gName,"",0)																// might also be an image plot (takes preference)
		trace = StringFromList(0, traceList)
		if (!strlen(trace))
			contour = ContourInfo(gName,StringFromList(0,ContourNameList(gName,";")),0)				// try with a contour trace at last
		endif
	endif
	
	int isContour = strlen(contour)
	int isImage   = strlen(image)
	
	if ((!strlen(trace) && !strlen(image) && !isContour) || !strlen(fitFunc) || !strlen(fitType))	// still no trace, image or function => abort
		Print "SQF error: No valid target or function."
		return -1
	endif
	
	// ++++++++++++++++++++++++ set up the input data and load options +++++++++++++++++++++++++++++
	Make/FREE/T/N=(4) YW_labels = ""
	Make/FREE/T/N=(4) XW_labels = ""
	s.pntMin  =  0; s.pntMax  = -1; s.xw_Min = -1; s.yw_trDim = 0; s.yw_inc = 1						// initialize all values
	s.pntMinY = -1; s.pntMaxY = -1; s.xw_Max = -1; s.xw_trDim = 0; s.xw_inc = 1
	s.fitFunc = fitFunc
	
	if (isMultiPeakFit)
		SplitString/E="([^ ]*) " s.fitFunc, fitFunc													// remove the number
	endif
	
	int isXYZcontour = 0
	if (isImage)
		WAVE s.data = $(StringByKey("ZWAVEDF",image) + PossiblyQuoteName(StringByKey("ZWAVE",image)))
		WAVE/Z s.xw = $(StringByKey("XWAVEDF",image) + PossiblyQuoteName(StringByKey("XWAVE",image)))
		WAVE/Z s.yw = $(StringByKey("YWAVEDF",image) + PossiblyQuoteName(StringByKey("YWAVE",image)))
		if (WaveDims(s.data)>2)
			return -1																				// higher dimension images not supported
		endif
		if (WaveExists(s.xw) || WaveExists(s.yw))													// no support for x/y waves with images
			return -100
		endif
		s.pntMinY = 0
		s.pntMaxY = DimSize(s.data,1)-1
		s.pntMax  = DimSize(s.data,0)-1
	elseif (isContour)
		WAVE s.data = $(StringByKey("ZWAVEDF",contour) + PossiblyQuoteName(StringByKey("ZWAVE",contour)))
		WAVE/Z s.xw = $(StringByKey("XWAVEDF",contour) + PossiblyQuoteName(StringByKey("XWAVE",contour)))
		WAVE/Z s.yw = $(StringByKey("YWAVEDF",contour) + PossiblyQuoteName(StringByKey("YWAVE",contour)))
		isXYZcontour = WaveExists(s.xw) && WaveExists(s.yw)
		s.pntMinY = isXYZcontour ? -1 : 0
		s.pntMaxY = DimSize(s.data,1)-1
		s.pntMax  = DimSize(s.data,0)-1
	else
		WAVE s.data = TraceNameToWaveRef(gName, trace)
		WAVE/Z s.xw =  XWaveRefFromTrace(gName, trace)
		getTraceYDimValues(s, gName, trace, YW_labels)												// fetch all dimension values
		getTraceXDimValues(s, gName, trace, XW_labels)
	endif
	variable yw_start = s.pntMin																	// local variants of the trace range (for later comparison)
	variable xw_start = s.xw_Min
	variable yw_end = s.pntMax
	variable xw_end = s.xw_Max
	
	string settings = StrVarOrDefault(localSettingsPath,SuperQuickFit#SQF_LoadPreferences())		// load settings
	string/G $localSettingsPath = settings															// save used settings to experiment
	int textboxOpt	= NumberByKey("textboxOpt",	 settings)
	int rangeMode	= NumberByKey("fitRangeMode",settings)
	int doWeight	= NumberByKey("weightError", settings)
	int doCleanUp	= NumberByKey("outCleanup",	 settings)
	
	int doFitAlign	= NumberByKey("plotAligned", settings);	doFitAlign  = (doFitAlign  <  0) ? 0 : doFitAlign		// these option were introduced later and may not be initialized yet
	int doMultiOut	= NumberByKey("multiColMode",settings);	doMultiOut  = (doMultiOut  <  0) ? 0 : doMultiOut
	int ForceStyle	= NumberByKey("trForceStyle",settings);	ForceStyle  = (ForceStyle  <  0) ? 0 : ForceStyle
	int SortBehind	= NumberByKey("trSortBehind",settings);	SortBehind  = (SortBehind  <  0) ? 0 : SortBehind
	int setFitColor	= NumberByKey("trColorMode", settings);	setFitColor = (setFitColor <  0) ? 0 : setFitColor
	int setFitStyle	= NumberByKey("trStyleMode", settings);	setFitStyle = (setFitStyle <  0) ? 0 : setFitStyle
	int FitLineSize	= NumberByKey("trLineSize",  settings);	FitLineSize = (FitLineSize <= 0) ? 1 : limit(FitLineSize,0,10)
	string FitColor = StringByKey("trColor",	 settings)
	
	s.plotFull	= NumberByKey("plotFull",  settings)
	s.doTextbox = NumberByKey("addTextBox",settings)
	s.holdMode  = NumberByKey("offsetMode",settings)
	s.holdStr 	= "1"																				// default is baseline hold
	
	// +++++++++++++++++++++++++++ extract cursor names and positions ++++++++++++++++++++++++++++++
	string csrA = ""
	string csrB = ""
	string csrA_trace = ""
	string csrB_trace = ""
	
	// cursor info: [0] wave x-value (attached)
	// [1] wave x point (attached) or relative axis range 0-1 (free)
	// [2] wave's x value (attached) or axis x value (free)
	// [3] wave's y value (attached) or axis y value (free)
	Make/FREE/D/N=4 csrA_vals = NaN
	Make/FREE/D/N=4 csrB_vals = NaN
	
	int i = 0
	for (; i < ItemsInList(validCursorNameList);)													// see which cursor is placed
		csrA = StringFromList(i++,validCursorNameList)
		if (strlen(CsrInfo($csrA,gName)))
			csrA_vals  = {xcsr($csrA, gName), pcsr($csrA, gName), hcsr($csrA, gName), vcsr($csrA, gName)}
			csrA_trace = StringByKey("TNAME",CsrInfo($csrA,gName))
			break
		endif
		csrA = ""
	endfor
	
	for (; i < ItemsInList(validCursorNameList);)													// poll the second cursor
		csrB = StringFromList(i++,validCursorNameList)
		if (strlen(CsrInfo($csrB,gName)))
			csrB_vals  = {xcsr($csrB, gName), pcsr($csrB, gName), hcsr($csrB, gName), vcsr($csrB, gName)}
			csrB_trace = StringByKey("TNAME",CsrInfo($csrB,gName))
			break
		endif
		csrB = ""
	endfor

	// +++++++++++++++++++++++++++++++++ find the fit range ++++++++++++++++++++++++++++++++++++++++
	variable xOff, xMul, yOff, yMul, csrXOff, csrXMul, tmpValue, xMin, xMax, yMin, yMax
	getAllTraceOffsets(gName, trace, xOff, xMul, yOff, yMul, (!isImage && !isContour))
	
	xMin = NaN; xMax = NaN; yMin = NaN; yMax = NaN
	switch(rangeMode)
		case 1:			// cursor range
			int pntCsr, isFree
			if (!strlen(csrA) || !strlen(csrB))														// both cursors must be on the graph
				break
			endif
			
			pntCsr = 0
			isFree = numtype(csrA_vals[0])
			if (!isFree && !isContour)
				if (isImage)
					pntCsr = WaveRefsEqual(s.data, CsrWaveRef($csrA,gName))							// cursor is attached to the same wave -> use points directly
				else
					pntCsr = !CmpStr(csrA_trace,trace)												// cursor is on the same trace
				endif
			endif
			
			if (pntCsr)
				s.pntMin = csrA_vals[1]
			else
				getAllTraceOffsets(gName, csrA_trace, csrXOff, csrXMul, tmpValue, tmpValue, (!isImage && !isContour && !isFree))
				xMin = ((csrA_vals[2]*csrXMul + csrXOff)-xOff)/xMul									// correct for scaling of both the source and possibly cursor traces
			endif
			
			pntCsr = 0
			isFree = numtype(csrB_vals[0])
			if (!isFree && !isContour)
				if (isImage)
					pntCsr = WaveRefsEqual(s.data, CsrWaveRef($csrB,gName))							// cursor is attached to the same wave -> use points directly
				else
					pntCsr = !CmpStr(csrB_trace,trace)												// cursor is on the same trace
				endif
			endif
			
			if (pntCsr)
				s.pntMax = csrB_vals[1]
			else
				getAllTraceOffsets(gName, csrB_trace, csrXOff, csrXMul, tmpValue, tmpValue, (!isImage && !isContour && !isFree))
				xMax = ((csrB_vals[2]*csrXMul + csrXOff)-xOff)/xMul
			endif
			
			if (s.pntMin == s.pntMax && numtype(xMin) && numtype(xMax))								// cursors at the same position => undo
				s.pntMin = yw_start
				s.pntMax = yw_end
			endif
			
			if (isImage || isContour)
				yMin = csrA_vals[3]
				yMax = csrB_vals[3]
			endif
		break
		case 2:			// axis range
			if (isImage)
				GetAxis/W=$gName/Q $StringByKey("XAXIS",image)
			else
				GetAxis/W=$gName/Q $StringByKey("XAXIS",TraceInfo(gName,trace,0))
			endif
			xMin = !V_flag ? V_min : xMin
			xMax = !V_flag ? V_max : xMax
			xMin = (xMin-xOff)/xMul																	// adjust for trace offsets
			xMax = (xMax-xOff)/xMul
			
			if (!isImage && !isContour)
				break
			endif
			
			if (isImage)
				GetAxis/W=$gName/Q $StringByKey("YAXIS",image)
			else
				GetAxis/W=$gName/Q $StringByKey("YAXIS",TraceInfo(gName,trace,0))
			endif
			yMin = !V_flag ? V_min : yMin
			yMax = !V_flag ? V_max : yMax
		break
		default:	// point range
		break
	endswitch
	
	WAVE/Z temp_xw = SQF_getTraceXWave(s)
	if ((!numtype(xMin) || !numtype(xMax)) && xMin != xMax)
		if (WaveExists(temp_xw))
			Duplicate/FREE temp_xw, pnt_xw; pnt_xw=p
			Sort temp_xw, temp_xw, pnt_xw															// make the x wave monotonic
			if (!numtype(xMin))
				FindLevel/P/Q temp_xw, xMin; s.pntMin = (!v_flag && v_levelX > yw_start && v_levelX < yw_end) ?  ceil(pnt_xw[v_levelX]) : s.pntMin
			endif
			if (!numtype(xMax))
				FindLevel/P/Q temp_xw, xMax; s.pntMax = (!v_flag && v_levelX > yw_start && v_levelX < yw_end) ? floor(pnt_xw[v_levelX]) : s.pntMax
			endif
		else
			if (!numtype(xMin))
				s.pntMin = limit(ScaleToIndex(s.data,xMin,s.yw_trDim), yw_start, yw_end)
			endif
			if (!numtype(xMax))
				s.pntMax = limit(ScaleToIndex(s.data,xMax,s.yw_trDim), yw_start, yw_end)
			endif
		endif
	endif
	if (!numtype(yMin) && !numtype(yMax) && !isXYZcontour)											// only for images and matrix contours
		s.pntMinY = limit(ScaleToIndex(s.data,yMin,1), 0, DimSize(s.data,1)-1)
		s.pntMaxY = limit(ScaleToIndex(s.data,yMax,1), 0, DimSize(s.data,1)-1)
	endif
	
	if (s.pntMin > s.pntMax)
		tmpValue = s.pntMin
		s.pntMin = s.pntMax
		s.pntMax = tmpValue
	endif
	if (s.pntMinY > s.pntMaxY)
		tmpValue  = s.pntMinY
		s.pntMinY = s.pntMaxY
		s.pntMaxY = tmpValue
	endif
	
	// +++++++++++++++++++++++++++++++++ find the baseline values ++++++++++++++++++++++++++++++++++
	variable xDelta = 1
	if (WaveExists(temp_xw))
		s.xBase = temp_xw[0]
		xDelta  = temp_xw[DimSize(temp_xw,0)-1]-temp_xw[0]
	else
		s.xBase = DimOffset(s.data,s.yw_trDim)
		xDelta  = DimDelta(s.data,s.yw_trDim)
	endif
	if (rangeMode == 1 && !numtype(xMin) && !numtype(xMax))
		s.xBase = ((xMin*sign(xDelta)) < (xMax*sign(xDelta))) ? xMin : xMax
	endif
	
	if (isImage || (isContour && !isXYZcontour))
		WaveStats/Q/RMD=[s.pntMin,s.pntMax][s.pntMinY,s.pntMaxY] s.data
	else
		WAVE/Z temp_yw = SQF_getTraceWave(s)														// something unknown went wrong
		if (!WaveExists(temp_yw))																	
			return -1000
		endif
		WaveStats/Q temp_yw
	endif
	s.yBase = (V_max > V_min) ? V_min : V_max
	
	// +++++++++++++++++++++++++++++++++ prepare the fit +++++++++++++++++++++++++++++++++++++++++++
	DebuggerOptions; int doDebug=V_debugOnError
	DebuggerOptions debugOnError=0
	
	int error = 0
	string tmpStr
	string cwPath = ""
	if (isUserFitFunc || isMultiPeakFit)
		FUNCREF dummyFunc_prepareCoef coefFunc=$(fitFunc + CoefGen_SUFFIX)							// prepare coef wave
		try
			error = coefFunc(s); AbortOnRTE
		catch
			error = GetRTError(1)
		endtry
		if (error || !WaveExists(s.cw))
			Print "Guess for " + fitFunc + " failed."
			DebuggerOptions debugOnError=doDebug
			return -10
		endif
		cwPath = GetWavesDataFolder(s.cw,2) + " "
		tmpStr = "•Make/D/O/N=" + num2str(DimSize(s.cw,0)) + " W_coef={"
		for (i=0; i<DimSize(s.cw,0); i++)
			tmpStr += num2str(s.cw[i]) + ","
		endfor
		tmpStr = RemoveEnding(tmpStr,",") + "}"
		Print tmpStr																				// print coefficient creation command
	endif
	
	int size
	int dStart = (s.pntMin - yw_start)/s.yw_inc														// calculate difference between selected and plotted range
	int dEnd   = (s.pntMax - yw_end)  /s.yw_inc
	
	if (dStart!=0 || dEnd!=0)																		// fit range changed -> need to reconstruct range labels
			size = DimSize(s.data, s.yw_trDim)-1
			YW_labels[s.yw_trDim]  = num2str(s.pntMin) + ","
			YW_labels[s.yw_trDim] += SelectString((s.pntMax==size), num2str(s.pntMax), "*")
			YW_labels[s.yw_trDim] += SelectString((s.yw_inc==1), ";" + num2str(s.yw_inc), "")
		if (strlen(XW_labels[s.xw_trDim]))
			size = DimSize(s.xw, s.xw_trDim)-1
			XW_labels[s.xw_trDim]  = num2str(xw_start + dStart*s.xw_inc) + ","
			XW_labels[s.xw_trDim] += SelectString(((xw_end + dEnd*s.xw_inc)==size), num2str(xw_end + dEnd*s.xw_inc), "*")
			XW_labels[s.xw_trDim] += SelectString((s.xw_inc==1), ";" + num2str(s.xw_inc), "")
		endif
	endif
	
	if (!CmpStr(YW_labels[0],"*") && !strlen(YW_labels[1] + YW_labels[2] + YW_labels[3]))			// range can be omitted here
		YW_labels = ""
	endif
	if (!CmpStr(XW_labels[0],"*") && !strlen(XW_labels[1] + XW_labels[2] + XW_labels[3]))
		XW_labels = ""
	endif
	
	size = DimSize(s.data,1)-1
	if ((isImage || (isContour && !isXYZcontour)) && (s.pntMinY>0 || s.pntMaxY<size || strlen(YW_labels[0])))
		YW_labels[0] = SelectString(strlen(YW_labels[0]),"*",YW_labels[0])							// need to have all range labels in this case
		YW_labels[1] = SelectString((s.pntMinY>0), "0", num2str(s.pntMinY)) + "," + SelectString((s.pntMaxY<size), "*", num2str(s.pntMaxY))
		YW_labels[1] = SelectString((s.pntMinY==0 && s.pntMaxY==size), YW_labels[1], "*")			// * means full range
	endif
	
	YW_labels = SelectString(strlen(YW_labels[p]), "", "[" + YW_labels[p] + "]")					// add brackets before insertion into command
	XW_labels = SelectString(strlen(XW_labels[p]), "", "[" + XW_labels[p] + "]")
	
	// +++++++++++++++++++++++++++++++ extract error-bars info +++++++++++++++++++++++++++++++++++++
	string regex, errInfo, eRow_lbl, eCol_lbl, eLay_lbl, eChk_lbl
	SplitString/E=("(?:.*)(?:wave=\()([^\)]*)\)\;RECREATION") TraceInfo(gName,trace,0), errInfo		// find error bar wave: finds contents of last wave=(CONTENT)
	
	string ebName = ""
	string xwName = ""
	string ywName = ""
	
	if (strlen(errInfo))
		regex  = SelectString(!CmpStr(errInfo[0],"'"), "([^\[|,]*)", "'?([^']*)'")					// name is liberal or not
		regex += "(?:\[(.*?)\])?(?:\[(.*?)\])?(?:\[(.*?)\])?(?:\[(.*?)\])?" + ",?(?:.*)"			// captures up to 4 range boxes
		SplitString/E=(regex) errInfo, ebName, eRow_lbl, eCol_lbl, eLay_lbl, eChk_lbl
		WAVE/T EB_labels = ListToTextWave(eRow_lbl + "#" + eCol_lbl + "#" + eLay_lbl + "#" + eChk_lbl + "#", "#")
		WAVE/Z ebWave = $ebName
		
		if ((dStart!=0 || dEnd!=0) && WaveExists(ebWave))											// adjust error-wave range
			variable err_0 = NaN
			variable err_1 = NaN
			variable estep = 1
			Extract/FREE/INDX EB_labels, eb_dim, StringMatch(EB_labels,"*,*") || !CmpStr(EB_labels,"*")
			[err_0, err_1, estep] = SQF_SplitRange(ebWave, eRow_lbl, eb_dim[0])						// split numbers in range
			size  = DimSize(ebWave,eb_dim[0])-1
			err_0 = !numtype(err_0) ? err_0 : 0
			err_1 = !numtype(err_1) ? err_1 : size
			EB_labels[eb_dim[0]]  = num2str(err_0 + dStart*estep) + ","
			EB_labels[eb_dim[0]] += SelectString(((err_1 + dEnd*estep)==size), num2str(err_1 + dEnd*estep), "*")
			EB_labels[eb_dim[0]] += SelectString((estep==1), ";" + num2str(estep), "")
		endif
		
		EB_labels = SelectString(strlen(EB_labels[p]), "", "[" + EB_labels[p] + "]")
	endif
	
	// ++++++++++++++++++++++++++++++++++++ do the fit +++++++++++++++++++++++++++++++++++++++++++++
	if (WaveExists(ebWave) && doWeight)																// add x-wave and error info
		sPrintf ebName, "/W=%s%s%s%s%s/I" ,GetWavesDataFolder(ebWave,2+2*(fitUseRelativePaths==1)), EB_labels[0], EB_labels[1], EB_labels[2], EB_labels[3]
	endif
		sPrintf ywName, "%s%s%s%s%s"	  ,GetWavesDataFolder(s.data,2+2*(fitUseRelativePaths==1)), YW_labels[0], YW_labels[1], YW_labels[2], YW_labels[3]
	if (isContour)
		if (WaveExists(s.xw) && WaveExists(s.yw))													// must be an XYZ contour
			sPrintf ywName,"%s"			  ,GetWavesDataFolder(s.data,2+2*(fitUseRelativePaths==1))
			sPrintf xwName,"/X={%s,%s}"	  ,GetWavesDataFolder(s.xw,  2+2*(fitUseRelativePaths==1)), GetWavesDataFolder(s.yw,2+2*(fitUseRelativePaths==1))
		endif
	else
		if (WaveExists(s.xw))
			sPrintf xwName,"/X=%s%s%s%s%s",GetWavesDataFolder(s.xw,  2+2*(fitUseRelativePaths==1)), XW_labels[0], XW_labels[1], XW_labels[2], XW_labels[3]
		endif
	endif
	
	string fitName = "fit_" + NameOfWave(s.data)
	WAVE/Z fit = $fitName
	if (WaveExists(fit))																			// clear note to append derived values later
		Note/K $fitName
	endif
	
	if (s.plotFull && (abs(xOff/xDelta) > (10^-12) || abs(xMul-1) > (10^-12)))
		Print "•SQF Caution: 'Plot Full Range' is active but trace " + trace + " is shifted. This may not work."
	endif
	
	string doTxtBox	= SelectString((s.doTextbox), "", "/TBOX=" + num2str(textboxOpt))				// construct the fit command
	string dofullR	= SelectString((s.plotFull),  "", "/X=1")
	string doHold	= SelectString((s.holdMode>0),"", "/H=\"" + s.holdStr + "\"")
	sPrintf tmpStr, "•%s/M=2%s%s%s %s, %s%s%s%s/D\r", fitType, doTxtBox, dofullR, doHold, fitFunc, cwPath, ywName, xwName, ebName
	Print tmpStr
	Execute/Z tmpStr																				// do the fit
	
	WAVE/Z s.fit = $fitName
	if (V_flag)
		Print "Execution of the fit failed."
		if (WaveExists(s.fit))																		// invalidate any previous fit curve
			s.fit = NaN
		endif
		return -999
	endif

	if (StringMatch(fitType, "CurveFit"))
		WAVE s.cw = W_coef
	endif
	if (StringMatch(fitFunc, "Line"))																// the line func does not output M_Covar
		WAVE s.sw = W_sigma
	else
		WAVE s.sw = M_Covar
	endif
	
	// +++++++++++++++++++++++++++++++++++ post processing +++++++++++++++++++++++++++++++++++++++++
	string validFunctions = FunctionList("*" + derivedVal_SUFFIX, ";", "KIND:2,VALTYPE:4")
	
	if (WhichListItem(fitFunc + derivedVal_SUFFIX, validFunctions) > -1)							// fit-function specific post processing
		FUNCREF dummyFunc_derivedVals outputFunc=$(fitFunc + derivedVal_SUFFIX)						// output additional parameters
		tmpStr = ""
		try
			tmpStr = outputFunc(s); AbortOnRTE
		catch
			Print "Error when executing " + fitFunc + derivedVal_SUFFIX + "(): " + GetRTErrMessage()
			error = GetRTError(1)
		endtry
		if (strlen(tmpStr))
			Print tmpStr
		endif
		if (WaveExists(s.fit))
			Note s.fit, tmpStr
		endif
	endif
	
	if (WhichListItem(genericPostprocName + derivedVal_SUFFIX, validFunctions) > -1)				// generic post-processing
		FUNCREF dummyFunc_derivedVals outputFunc=$(genericPostprocName + derivedVal_SUFFIX)
		tmpStr = ""
		try
			tmpStr = outputFunc(s); AbortOnRTE
		catch
			Print "Error when executing " + genericPostprocName + derivedVal_SUFFIX + "(): " + GetRTErrMessage()
			error = GetRTError(1)
		endtry
		if (strlen(tmpStr))
			Print tmpStr
		endif
		if (WaveExists(s.fit))
			Note s.fit, tmpStr
		endif
	endif
	
	DebuggerOptions debugOnError=doDebug
	
	// +++++++++++++++++++++++++++++++++++++++ cleanup +++++++++++++++++++++++++++++++++++++++++++++
	
	string remList = ""
	if (doCleanUp)																					// remove working variables and waves
		remList  = "V_FitOptions;V_FitTol;V_tol;V_chisq;V_q;V_siga;V_sigb;V_Rab;V_Pr;V_r2;V_npnts;V_nterms;V_nheld;V_numNaNs;V_numINFs;"
		remList += "V_FitError;V_FitQuitReason;V_FitIterStart;V_FitMaxIters;V_FitNumIters;"
		remList += "V_startRow;V_startCol;V_startLayer;V_startChunk;V_endRow;V_endCol;V_endLayer;V_endChunk;"
		for(i=0; i<ItemsInList(remList); i++)
			Killvariables/Z $StringFromList(i,remList)
		endfor
		remList = "S_Info;"
		for(i=0; i<ItemsInList(remList); i++)
			Killstrings/Z $StringFromList(i,remList)
		endfor
		KillWaves/Z W_coef, W_sigma, M_Covar, W_ParamConfidenceInterval, W_fitConstants
	endif
	
	if (!WaveExists(s.fit) || isImage || isContour)
		return 0
	endif
	
	// +++++++++++++++++++++++++++++++++ trace related options +++++++++++++++++++++++++++++++++++++
	
	string fitTrace  = PossiblyQuoteName(NameOfWave(s.fit))
	string fitXTrace = ""
	int isPlotted	 = FindListItem(fitTrace, TraceNameList(gName,";",1))!=-1
	int wasPlotted	 = FindListItem(fitTrace, traceList)!=-1
	
	if (isPlotted)
		fitXTrace = PossiblyQuoteName(StringByKey("XWAVE",TraceInfo(gName, fitTrace, 0)))
		if (strlen(fitXTrace))
			fitXTrace[0] = " vs "
		endif
	endif
	
	if (strlen(YW_labels[1]) && s.yw_trDim==0 && doMultiOut)										// ver. 1.09 - create individual fit for each column
		string newFitName
		int colDigits = strlen(num2str(DimSize(s.data,1)-1))										// the number of digits is decided by the order of magnitude of the size
		sPrintf newFitName, "fit_%.*d_%s", colDigits, s.yw_dims[1], NameOfWave(s.data)
		RemoveFromGraph/Z/W=$gName $fitTrace
		Duplicate/O s.fit, $newFitName; KillWaves/Z $fitName
		fitTrace   = PossiblyQuoteName(newFitName)
		isPlotted  = FindListItem(fitTrace, TraceNameList(gName,";",1)) != -1
		wasPlotted = FindListItem(fitTrace, traceList) != -1
		
		// +++++++++++++++++++++++++ make the results text box unique ++++++++++++++++++++++++++++++
		string clnName = CleanupName(NameOfWave(s.data),0)
		string newName; sPrintf newName, "CF_%.*d_%s", colDigits, s.yw_dims[1], clnName
		string curText = ""
		string curFlag = "/N=" + newName
		string curName = "CF_" + clnName
		
		if (WhichListItem(newName, AnnotationList(gName)) > -1)
			curFlag = StringByKey("FLAGS",AnnotationInfo(gName,newName,0))
			TextBox/W=$gName/N=$newName/K
		endif
		
		if (WhichListItem(curName, AnnotationList(gName)) > -1)										// ver. 1.12 - rename results text box to a distinct name
			curText = StringByKey("TEXT",AnnotationInfo(gName,curName,0))
			curText = ReplaceString(fitName,curText,newFitName)										// honor new fit trace name
			curText = ReplaceString("\s(" + PossiblyQuoteName(NameOfWave(s.data)) + ")", curText, "\s(" + trace + ")")	// make sure the data trace is correct
			sPrintf tmpStr, "TextBox/W=%s%s \"%s\"", gName, curFlag, curText
			TextBox/W=$gName/N=$("CF_" + clnName)/K
			Execute/Z/Q tmpStr
		endif
	endif
	
	string fitTraceFlags = StringByKey("AXISFLAGS",TraceInfo(gName, fitTrace, 0))					// ver. 1.09 - put fit trace on the same axes as source trace
	string inpTraceFlags = StringByKey("AXISFLAGS",TraceInfo(gName, trace, 0))
	if (CmpStr(fitTraceFlags,inpTraceFlags) || !isPlotted)											// if the source is on a different axis combination, re-add fit trace to the same axes
		RemoveFromGraph/Z/W=$gName $fitTrace
		sPrintf tmpStr, "AppendToGraph/W=%s%s %s%s", gName, inpTraceFlags, fitTrace, fitXTrace
		Execute/Z/Q tmpStr
		isPlotted = 1
	endif
	
	if (isPlotted)
		if (!wasPlotted || ForceStyle)
			sPrintf tmpStr, "ModifyGraph/W=%s lstyle(%s)=%d, lsize(%s)=%g", gName, fitTrace, setFitStyle, fitTrace, FitLineSize
			string colorMode = ""
			string inTrColor = ""
			switch(setFitColor)	// 0 is default
				case 4:			// fallthrough, lower saturation
					colorMode = SelectString(strlen(colorMode), "lowerSaturation", colorMode)
				case 3:			// fallthrough, complementary color
					colorMode = SelectString(strlen(colorMode), "complementColor", colorMode)
				case 2:			// same as input trace
					inTrColor = StringByKey("rgb(x)", TraceInfo(gName,trace,0), "=")
					inTrColor = ReplaceString("(", RemoveEnding(inTrColor,")"), "")					// remove outer brackets
					inTrColor = SuperQuickFit#RGB_modify(colorMode, inTrColor)
					sPrintf tmpStr, "%s ,rgb(%s)=(%s)", tmpStr, fitTrace, inTrColor
				break
				case 1:			// fixed color
					sPrintf tmpStr, "%s ,rgb(%s)=(%s)", tmpStr, fitTrace, FitColor
				break
				default:
				break
			endswitch
			Execute/Z/Q tmpStr
		endif
		
		if (doFitAlign)																				// ver. 1.06 - added option for aligned fit traces
			xMul = (xMul==1) ? 0 : xMul
			yMul = (yMul==1) ? 0 : yMul
			ModifyGraph/Z/W=$gName offset($fitTrace)={xOff,yOff}, muloffset($fitTrace)={xMul,yMul}
		endif
		
		if (SortBehind)
			ReorderTraces/W=$gName $trace,{$fitTrace}
		endif
	endif
	
	return 0
End

// *** trace wave constructor *** ##################################################################

Function/WAVE SQF_getTraceWave(STRUCT SuperQuickFitStruct &s)										// multi-dimension aware extraction of the plotted range from the input data

	if (!WaveExists(s.data))
		return $""
	endif
	
	switch(s.yw_trDim)
		case 0:
			Duplicate/FREE/R=[s.pntMin,s.pntMax][s.yw_dims[1]][s.yw_dims[2]][s.yw_dims[3]] s.data, out
		break
		case 1:
			Duplicate/FREE/R=[s.yw_dims[0]][s.pntMin,s.pntMax][s.yw_dims[2]][s.yw_dims[3]] s.data, out
		break
		case 2:
			Duplicate/FREE/R=[s.yw_dims[0]][s.yw_dims[1]][s.pntMin,s.pntMax][s.yw_dims[3]] s.data, out
		break
		case 3:
			Duplicate/FREE/R=[s.yw_dims[0]][s.yw_dims[1]][s.yw_dims[2]][s.pntMin,s.pntMax] s.data, out
		break
		default:
			return $""
		break
	endswitch
	
	Redimension/N=(abs(s.pntMax-s.pntMin),0,0,0) out
	if (s.yw_inc != 1)																				// remove points via increment
		Extract/O out, out, !mod(p,s.yw_inc)
	endif
	if (!DimSize(out,0))
		return $""
	endif
	
	variable off = DimOffset(s.data,s.yw_trDim)
	variable  dx = DimDelta( s.data,s.yw_trDim)
	SetScale/P x, off+dx*s.pntMin, dx*s.yw_inc, WaveUnits(s.data,s.yw_trDim), out					// transfer the scale for the reduced points
	
	return out
End

Function/WAVE SQF_getTraceXWave(STRUCT SuperQuickFitStruct &s)

	if (!WaveExists(s.xw) || s.xw_Min < 0 || s.xw_Max < 0)
		return $""
	endif
	
	switch(s.xw_trDim)
		case 0:
			Duplicate/FREE/R=[s.xw_Min,s.xw_Max][s.xw_dims[1]][s.xw_dims[2]][s.xw_dims[3]] s.xw, out
		break
		case 1:
			Duplicate/FREE/R=[s.xw_dims[0]][s.xw_Min,s.xw_Max][s.xw_dims[2]][s.xw_dims[3]] s.xw, out
		break
		case 2:
			Duplicate/FREE/R=[s.xw_dims[0]][s.xw_dims[1]][s.xw_Min,s.xw_Max][s.xw_dims[3]] s.xw, out
		break
		case 3:
			Duplicate/FREE/R=[s.xw_dims[0]][s.xw_dims[1]][s.xw_dims[2]][s.xw_Min,s.xw_Max] s.xw, out
		break
		default:
			return $""
		break
	endswitch
	
	Redimension/N=(abs(s.xw_Max-s.xw_Min),0,0,0) out
	if (s.xw_inc != 1)
		Extract/O out, out, !mod(p, s.xw_inc)
	endif
	if (!DimSize(out,0))
		return $""
	endif
	
	return out
End

//##################################################################################################

static Function getTraceYDimValues(STRUCT SuperQuickFitStruct &s, string gName, string trace, WAVE/T labels)

	variable r0, r1, inc, i; string str = ""
	for (i = 0; i < 4; i++)
		[r0, r1, inc]  = getRange(gName, trace, i, "Y", str)
		labels[i]      = str
		s.yw_dims[i]   = numType(r0) ? 0 : r0
		if (StringMatch(str,"*,*") || !CmpStr(str,"*"))												// this is the used dimension by the trace
			s.pntMin   = r0
			s.pntMax   = r1
			s.yw_inc   = inc
			s.yw_trDim = i
		endif
	endfor
End

static Function getTraceXDimValues(STRUCT SuperQuickFitStruct &s, string gName, string trace, WAVE/T labels)

	variable r0, r1, inc, i; string str = ""
	for (i = 0; i < 4; i++)
		[r0, r1, inc]  = getRange(gName, trace, i, "X", str)
		labels[i]      = str
		s.xw_dims[i]   = numType(r0) ? 0 : r0
		if (StringMatch(str,"*,*") || !CmpStr(str,"*"))
			s.xw_Min   = r0
			s.xw_Max   = r1
			s.xw_inc   = inc
			s.xw_trDim = i
		endif
	endfor
	if (WaveExists(s.xw))																			// make sure these are valid anyway
		s.xw_Min = (s.xw_Min>=0) ? s.xw_Min : 0
		s.xw_Max = (s.xw_Max>=0) ? s.xw_Max : (DimSize(s.xw,s.xw_trDim)-1)
	endif
End

//--------------------------------------------------------------------------------------------------

static Function [variable r0, variable r1, variable inc] getRange(string gName, string trace, int dim, string type, string &range)	// extract range from trace info

	r0 = NaN; r1 = NaN; inc = 1; range = ""
	if (dim < 0 || dim > 3 || WhichListItem(type,"X;Y;")<0)
		return [r0, r1, inc]
	endif
	if (!CmpStr(type,"X"))
		WAVE/Z data = XWaveRefFromTrace(gName, trace)
	endif
	if (!CmpStr(type,"Y"))
		WAVE/Z data = TraceNameToWaveRef(gName, trace)
	endif
	if (!WaveExists(data))
		return [r0, r1, inc]
	endif
	if (dim > (WaveDims(data)-1))
		return [r0, r1, inc]
	endif
	
	string expr = "(?:\[(.*?)\])?(?:\[(.*?)\])?(?:\[(.*?)\])?(?:\[(.*?)\])?"
	string info = StringByKey(type + "RANGE", TraceInfo(gName, trace, 0))
	string str0, str1, str2, str3
	SplitString/E=(expr) info, str0, str1, str2, str3												// find the displayed range from TraceInfo
	range = ReplaceString(":",StringFromList(dim, str0+";"+str1+";"+str2+";"+str3+";"), ";")		// range inc must be [a,b;c] instead of [a,b:c]
	[r0, r1, inc] = SQF_SplitRange(data, range, dim)
	
	return [r0, r1, inc]
End

//--------------------------------------------------------------------------------------------------

static Function [variable r0, variable r1, variable inc] SQF_SplitRange(WAVE data, string range, int dim)	// splits a trace's range dimension into start, end and increment values

	r0 = NaN; r1 = NaN; inc = 1
	if (!WaveExists(data) || !strlen(range))
		return [r0, r1, inc]
	endif
	
	string str0 = range
	string str1 = ""
	string step = ""
	if (StringMatch(range,"*,*"))																	// start/end range
		SplitString/E=("(.*),([^;]*);?(.*)") range, str0, str1, step								// split into first and second number and optionally a inc value
	endif
	if (!CmpStr(str0[0],"%"))																		// is tag
		r0 = FindDimLabel(data, dim, str0[1,inf])
		r0 = (r0 == -2) ? NaN : r0
	else
		r0 = !CmpStr(str0[0],"*") ? 0 : str2num(str0) 
	endif
	if (!CmpStr(str1[0],"%"))
		r1 = FindDimLabel(data, dim, str1[1,inf])
		r1 = (r1 == -2) ? NaN : r1
	else
		r1 = (!CmpStr(str1[0],"*") || !CmpStr(range,"*")) ? (DimSize(data,dim)-1) : str2num(str1)
	endif
	inc = str2num(step)
	inc = numtype(inc) ? 1 : inc
	
	return [r0, r1, inc]
End

// *** various helper functions *** ################################################################

static Function getAllTraceOffsets(string gName, string trace, variable &xOff, variable &xMul, variable &yOff, variable &yMul, int isTrace)

	xOff = 0; xMul = 1; yOff = 0; yMul = 1
	if (!isTrace)																					// just initialize
		return 0
	endif
	sscanf StringByKey("offset(x)",   TraceInfo(gName,trace,0), "="), "{%f,%f}", xOff, yOff
	sscanf StringByKey("muloffset(x)",TraceInfo(gName,trace,0), "="), "{%f,%f}", xMul, yMul
	xMul = (xMul==0) ? 1 : xMul
	yMul = (yMul==0) ? 1 : yMul
End

//##################################################################################################

static Function SQF_GetUserFuncType(string func)													// finds function type of user function: 0 = invalid, 1 = 1D, 2 = 2D

	int type = 0
	string info = FunctionInfo(func)
	variable npara = NumberByKey("N_PARAMS",info)
	variable para1 = NumberByKey("PARAM_1_TYPE",info)
	variable para2 = NumberByKey("PARAM_2_TYPE",info)
	switch(npara)
		case 2:
			if (para1 & 4)
				type = 1
			endif
		break
		case 3:
			if ((para1 & 0x4000) && (para2 & 0x4000))												// all-at-once function
				type = 1
			elseif ((para1 & 4) && (para2 & 4))
				type = 2
			endif
		break
		case 4:
			if ((para1 & 0x4000) && (para2 & 0x4000))												// all-at-once function
				type = 2
			endif
		break
		default:
		break
	endswitch
	
	return type
End

//##################################################################################################

static Function/S RGB_modify(String which, String color)

	int R = str2num(StringFromList(0,color,",")); R = (R <  0) ? 0 : R
	int G = str2num(StringFromList(1,color,",")); G = (G <  0) ? 0 : G
	int B = str2num(StringFromList(2,color,",")); B = (B <  0) ? 0 : B
	int A = str2num(StringFromList(3,color,",")); A = (A <= 0) ? 65535 : A
	string returnColor = ""
	strswitch(which)
		case "complementColor":
				R = 65535-R; G = 65535-G; B = 65535-B;												// calculate complementary colors
			if (min(R,G,B) >= (235*257))															// color is too bright to see
				R -= 15*257; G -= 15*257; B -= 15*257;												// make a bit darker
			endif
		break
		case "lowerSaturation":
			variable H,S,V
			[H,S,V] = SuperQuickFit#RGB2HSV(R,G,B)													// extract and reduce saturation
			 S = limit(S-0.75, 0, 1)																// reduce saturation
			 V = limit((!S ? ((V + 1)/2) : (V-0.15)), 0, 1)											// is already gray => make lighter instead
			[R,G,B] = SuperQuickFit#HSV2RGB(H,S,V)
		break
		default:
		break
	endswitch
	
	sPrintf returnColor, "%d,%d,%d,%d", R, G, B, A
	return returnColor
End

//##################################################################################################

static Constant kBitDepth = 65535																	// 16 bit colors

static Function [variable H, variable S, variable V] RGB2HSV(int R, int G, int B)					// assumes R,G,B values in the range [0..65535]
																									// outputs H = [0..360], S,V = [0..1]
	variable Mx = max(R,G,B)
	variable Mn = min(R,G,B)
	H = acos((R-G/2-B/2)/sqrt(R^2+G^2+B^2-R*G-R*B-G*B))*180/pi
	H = (G >=B)    ? (360-H) : H
	H = numtype(H) ? 0 : H
	S = (Mx==0)    ? 0 : (1-Mn/Mx)
	V = Mx/kBitDepth
End

static Function [int R, int G, int B] HSV2RGB(variable H, variable S, variable V)					// assumes a H value in the range [0..360] and S,V values in the range [0..1]
																									// outputs R,G,B = [0..65535]
	variable C = V*kBitDepth
	variable M = (1-S)*C
	variable X = (C-M)*(1-abs(mod(H/60,2)-1))
	if     (H >=   0 && H < 60)
		R = C; G = M; B = X+M;
	elseif (H >=  60 && H < 120)
		R = X+M; G = M; B = C;
	elseif (H >= 120 && H < 180)
		R = M; G = X+M; B = C;
	elseif (H >= 180 && H < 240)
		R = M; G = C; B = X+M;
	elseif (H >= 240 && H < 300)
		R = X+M; G = C; B = M;
	elseif (H >= 300 && H <= 360)
		R = C; G = X+M; B = M;
	endif
End

//__________________________________________________________________________________________________
// *** output definitions for default fit-functions ***
//__________________________________________________________________________________________________

Function/S Line_derivedVals(STRUCT SuperQuickFitStruct &s)

	Variable abCor = NumVarOrDefault("V_rab",0)
	variable x_crs = -s.cw[0]/s.cw[1]
	variable x_err = x_crs * sqrt( (s.sw[0]/s.cw[0])^2 + (s.sw[1]/s.cw[1])^2 + 2*(abCor*s.sw[0]*s.sw[1])/(s.cw[0]*s.cw[1]) )
	string str;	sPrintf str, "Derived values:\r\tX-intercept\t= %g ± %g", x_crs, (numtype(x_err) ? 0 : x_err)
	if (s.doTextbox)
		AppendText "\t" + str
	endif
	
	return str
End

Function/S Gauss_derivedVals(STRUCT SuperQuickFitStruct &s)

	variable pA	= abs(s.cw[3]) * sqrt(Pi) * s.cw[1]
	variable pW	= abs(s.cw[3])*2*sqrt(ln(2))
	variable pA_err	= sqrt((s.sw[3][3]/s.cw[3]^2) + (s.sw[1][1]/s.cw[1]^2) + 2*s.sw[1][3]/(s.cw[1]*s.cw[3])) * pA
	variable pW_err	= sqrt( s.sw[3][3] )*2*sqrt(ln(2))
	
	string str;	sPrintf str, "Derived values:\r\tArea\t= %g ± %g\r\tFWHM\t= %g ± %g", pA, pA_err, pW, pW_err
	if (s.doTextbox)
		AppendText "\t" + str
	endif
	
	return str
End

Function/S Lor_derivedVals(STRUCT SuperQuickFitStruct &s)

	variable pH	= s.cw[1]/abs(s.cw[3])
	variable pW	= 2*sqrt(abs(s.cw[3]))
	variable pA	= pi*s.cw[1]/sqrt(abs(s.cw[3]))
	variable pH_err	= sqrt((s.sw[3][3]/s.cw[3]^2) + (s.sw[1][1]/s.cw[1]^2) + 2*s.sw[1][3]/(s.cw[1]*s.cw[3])) * pH
	variable pW_err	= sqrt( s.sw[3][3] ) * pW/s.cw[3]
	variable pA_err	= sqrt( s.sw[1][1] * pi^2/abs(s.cw[3]) + s.sw[3][3] * (pi^2*s.cw[1]^2)/(4*abs(s.cw[3])^3) - 2*abs(s.sw[1][3])*(pi*s.cw[1])/(2*abs(s.cw[3])^2) )
	
	string str;	sPrintf str, "Derived values:\r\tHeight\t= %g ± %g\r\tArea\t= %g ± %g\r\tFWHM\t= %g ± %g", pH, pH_err, pA, pA_err, pW, pW_err
	if (s.doTextbox)
		AppendText "\t" + str
	endif
	
	return str
End

Function/S Voigt_derivedVals(STRUCT SuperQuickFitStruct &s)

	variable C1 = 0.5346																			// empirical constants
	variable C2 = 0.2166
	variable k  = 2*sqrt(ln(2)/pi)
	variable pS	=   s.cw[4] * sqrt(ln(2))															// peak shape
	variable pH	= k*s.cw[1] / abs(s.cw[3]) * exp(pS^2) * erfc(pS)
	variable pW	= (C1 * s.cw[4] + sqrt(C2 * s.cw[4]^2 + 1))*abs(s.cw[3])
	
	variable ExpErf = exp(pS^2)*erfc(pS)
	variable piXErf = sqrt(pi)*pS*ExpErf
	variable abscw3 = abs(s.cw[3])
	
	variable pH_err = 0
	pH_err +=   s.sw[1][1] * (k / abscw3 * ExpErf)^2
	pH_err +=   s.sw[2][2] * ((piXErf-1) * k^2 * s.cw[1]/abscw3)^2
	pH_err +=   s.sw[3][3] * (k   * s.cw[1]/abscw3^2 *  ExpErf)^2
	pH_err += 2*s.sw[1][3] *  k^2 * s.cw[1]/abscw3^3 * (ExpErf)^2
	pH_err -= 2*s.sw[1][2] * (piXErf-1) * k^3 * (s.cw[1]/abscw3)   * ExpErf
	pH_err -= 2*s.sw[2][3] * (piXErf-1) * k^3 * (s.cw[1]/abscw3)^2 * ExpErf
	pH_err  = sqrt(pH_err)
	
	variable pW_err = 0
	pW_err +=		s.sw[3][3] * (  C1*s.cw[4] + sqrt(C2*s.cw[4]^2 + 1) )^2
	pW_err +=		s.sw[4][4] * ( (C1*s.cw[4] + 	  C2*s.cw[4] / sqrt(  C2*s.cw[4]^2 + 1) ) * abs(s.cw[3]) )^2
	pW_err += 2*abs(s.sw[3][4])* (  C1*s.cw[4] + sqrt(C2*s.cw[4]^2 + 1))*(C1*s.cw[4] + C2*s.cw[4]/sqrt(C2*s.cw[4]^2 + 1))*abs(s.cw[3])
	pW_err  = sqrt(pW_err)
	
	string str;	sPrintf str, "Derived values:\r\tHeight\t= %g ± %g\r\tFWHM\t= %g ± %g", pH, pH_err, pW, pW_err
	if (s.doTextbox)
		AppendText "\t" + str
	endif
	
	return str
End

//__________________________________________________________________________________________________
// *** custom fit functions ***
//__________________________________________________________________________________________________

Function FermiEdge_Line(WAVE w, variable E) : FitFunc

	//CurveFitDialog/ Equation:
	//CurveFitDialog/ f(E) = y0 + (m*E + A) / (exp((E-Ef)/kT) + 1)
	//CurveFitDialog/ 
	//CurveFitDialog/ End of Equation
	//CurveFitDialog/ Independent Variables 1
	//CurveFitDialog/ E
	//CurveFitDialog/ Coefficients 5
	//CurveFitDialog/ w[0] = y0
	//CurveFitDialog/ w[1] = Ef
	//CurveFitDialog/ w[2] = kT
	//CurveFitDialog/ w[3] = A
	//CurveFitDialog/ w[4] = m
	
	return w[0] + (w[4]*E + w[3]) / (exp((E-w[1])/w[2]) + 1)
End

Function FermiEdge_Line_prepareCoef(STRUCT SuperQuickFitStruct &s)

	WAVE   temp_yw = SQF_getTraceWave(s)
	WAVE/Z temp_xw = SQF_getTraceXWave(s)
	variable Ef
	variable wMax = WaveMax(temp_yw)
	variable smth = limit(round(abs(s.pntMax-s.pntMin)/25), 1, 2^15-1)
	
	Differentiate temp_yw/D=temp_yw
	Smooth smth, temp_yw
	if (WaveExists(temp_xw))
		WaveStats/Q/P temp_yw
		Ef = temp_xw[V_minRowLoc]
	else
		WaveStats/Q   temp_yw
		Ef = V_minloc
	endif
	
	Make/D/O/N=2 W_coef={((s.holdMode==1) ? 0 : s.yBase), Ef, 0.025, wMax, -1}
	WAVE s.cw = W_coef
	
	return 0	// no error
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

Function LineX(WAVE w, variable x) : FitFunc

	//CurveFitDialog/ Equation:
	//CurveFitDialog/ f(x) = m*(x-x0)
	//CurveFitDialog/ End of Equation
	//CurveFitDialog/ Independent Variables 1
	//CurveFitDialog/ x
	//CurveFitDialog/ Coefficients 2
	//CurveFitDialog/ w[0] = x0
	//CurveFitDialog/ w[1] = m
	
	return w[1]*(x-w[0])
End

Function LineX_prepareCoef(STRUCT SuperQuickFitStruct &s)

	WAVE temp_yw = SQF_getTraceWave(s)
	WaveStats/Q temp_yw;	variable Wmax = V_max; variable Wmin = V_min
	Differentiate temp_yw/D=temp_yw
	WaveStats/Q temp_yw;	variable slope = (Wmax > Wmin) ? V_max : V_min
	Make/D/O/N=2 W_coef = {((s.holdMode==1) ? 0 : s.xBase),slope}
	WAVE s.cw = W_coef
	
	return 0	// no error
End

Function/S LineX_derivedVals(STRUCT SuperQuickFitStruct &s)

	variable y_cross = -s.cw[1]*s.cw[0]
	variable y_error = y_cross * sqrt( (s.sw[0][0]/s.cw[0]^2) + (s.sw[1][1]/s.cw[1]^2) + 2*s.sw[0][1]/(s.cw[0]*s.cw[1]) )
	string str;	sPrintf str, "Derived values:\r\tY-intercept\t= %g ± %g", y_cross, (numtype(y_error) ? 0 : y_error)
	if (s.doTextbox)
		AppendText "\t" + str
	endif
	
	return str
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

Function PAD_Beta(WAVE w, variable x) : FitFunc

	//CurveFitDialog/ Equation:
	//CurveFitDialog/ f(theta) = scale*(1 + beta*legendreA(2,0,cos((theta-theta_0)*(Pi/180))))
	//CurveFitDialog/ End of Equation
	//CurveFitDialog/ Independent Variables 1
	//CurveFitDialog/ theta
	//CurveFitDialog/ Coefficients 3
	//CurveFitDialog/ w[0] = scale
	//CurveFitDialog/ w[1] = beta
	//CurveFitDialog/ w[2] = theta_0
	
	return w[0]*(1 + w[1]*legendreA(2,0,cos((x-w[2])*(Pi/180))))
End

Function PAD_Beta_prepareCoef(STRUCT SuperQuickFitStruct &s)

	WAVE temp_yw = SQF_getTraceWave(s)
	WaveStats/Q temp_yw
	Make/D/O/N=3 W_coef = {V_max/3,2,0}
	WAVE s.cw = W_coef
	s.holdStr = "001"
	
	return 0	// no error
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

Function MultiGauss(WAVE w, variable x) : FitFunc

	variable numPeak = round((DimSize(w,0)-1)/3)
	if (numPeak < 1 || mod(DimSize(w,0)-1,3))
		return NaN
	endif
	variable val = 0; int i
	for (i = 0; i < numPeak; i++)
		val += w[3*i+1]*Gauss(x, w[3*i+2], w[3*i+3])
	endfor
	
	return w[0] + val
End

Function MultiGauss_prepareCoef(STRUCT SuperQuickFitStruct &s)

	WAVE   temp_yw = SQF_getTraceWave(s)
	WAVE/Z temp_xw = SQF_getTraceXWave(s)
	WAVE  peakList = SuperQuickFit#doFindPeaks(temp_yw,s.pntMin,s.pntMax)
	
	string str; SplitString/E=" ([0-9])" s.fitFunc, str
	variable numPeak = str2num(str)
	if (DimSize(peakList,0) < numPeak || numtype(numPeak))
		return -1	// error
	endif
	
	Redimension/N=(numPeak,-1) peakList																// sort x values
	Make/D/N=(numPeak)/FREE sortWave = peakList[p][0]
	MakeIndex sortWave, sortWave
	Duplicate/FREE peakList, peakListCP
	peakList = peakListCP[sortWave[p]][q]
	
	int i; int j = 1
	variable widthMod = 0.9
	
	Make/D/O/N=1 W_coef = {(s.holdMode == 1) ? 0 : s.yBase}
	for (i = 0; i < numPeak; i++)
		W_coef[j++] = {peakList[i][2]}
		if (WaveExists(temp_xw))
			W_coef[j++] = {temp_xw(peakList[i][0])}
			W_coef[j++] = {peakList[i][1] * widthMod * abs( (temp_xw[numpnts(temp_xw)-1]-temp_xw[0])/(numpnts(temp_xw)-1) )}
		else
			W_coef[j++] = {peakList[i][0] * DimDelta(temp_yw,0) + DimOffset(temp_yw,0)}
			W_coef[j++] = {peakList[i][1] * widthMod * abs(DimDelta(temp_yw,0))*0.8}
		endif
	endfor
	WAVE s.cw = W_coef
	
	return 0	// no error
End

Function/S MultiGauss_derivedVals(STRUCT SuperQuickFitStruct &s)

	int numPeak = round((DimSize(s.cw,0)-1)/3); int i
	if (numPeak < 1 || mod(DimSize(s.cw,0)-1,3))
		return ""
	endif
	
	string str = ""
	for (i = 0; i < numPeak; i++)
		int pA = 3*i+1
		int pL = 3*i+2
		int pW = 3*i+3
		variable FWHM  = 2*sqrt(2*ln(2)) * abs( s.cw[pW])
		variable W_err = 2*sqrt(2*ln(2)) * sqrt(s.sw[pW][pW])
		variable AMPL  = s.cw[pA]/( abs(s.cw[pW])*sqrt(2*Pi) )
		variable A_err = sqrt( (s.sw[pW][pW]/s.cw[pW]^2) + (s.sw[pA][pA]/s.cw[pA]^2) + 2*s.sw[pA][pW]/(s.cw[pA]*s.cw[pW]) ) * AMPL
		sPrintf str, "%s\tGauss %d: Area = K%d; Loc = K%d; Width = K%d, Height = %#g ± %#g, FWHM = %#g ± %#g\r", str, i+1, pA, pL, pW, AMPL, A_err, FWHM, W_err
	endfor
	
	str[0] = "Peak values:\r"
	if (s.doTextbox)
		AppendText "\t" + ReplaceString(";", ReplaceString(", ", str, "\r\t\t"), ",")
	endif
	
	return ReplaceString(";", str, ",")
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

Function DoubleGauss(WAVE w, variable x) : FitFunc

	//CurveFitDialog/ Equation:
	//CurveFitDialog/ f(x) = y0 + Area1 * Gauss(x,Loc1,Width1) + Area2 * Gauss(x,Loc2,Width2)
	//CurveFitDialog/ End of Equation
	//CurveFitDialog/ Independent Variables 1
	//CurveFitDialog/ x
	//CurveFitDialog/ Coefficients 7
	//CurveFitDialog/ w[0] = y0
	//CurveFitDialog/ w[1] = Area1
	//CurveFitDialog/ w[2] = Area2
	//CurveFitDialog/ w[3] = Loc1
	//CurveFitDialog/ w[4] = Loc2
	//CurveFitDialog/ w[5] = Width1
	//CurveFitDialog/ w[6] = Width2
	
	return w[0] + w[1]*Gauss(x,w[3],w[5]) + w[2]*Gauss(x,w[4],w[6])
End

Function DoubleGauss_prepareCoef(STRUCT SuperQuickFitStruct &s)

	WAVE   temp_yw = SQF_getTraceWave(s)
	WAVE/Z temp_xw = SQF_getTraceXWave(s)
	variable wMin = (s.holdMode == 1) ? 0 : s.yBase
	WAVE peakList = SuperQuickFit#doFindPeaks(temp_yw,s.pntMin,s.pntMax)
	if (DimSize(peakList,0) < 2)
		return -1	// error
	endif
	
	variable pL1 = peakList[0][0]
	variable pL2 = peakList[1][0]
	variable pW1 = peakList[0][1]
	variable pW2 = peakList[1][1]
	variable pH1 = peakList[0][2]
	variable pH2 = peakList[1][2]
	if (WaveExists(temp_xw))
		int xSize = DimSize(temp_xw,0)-1
		pL1 = temp_xw[pL1]
		pL2 = temp_xw[pL2]
		pW1 *= abs((temp_xw[xSize]-temp_xw[0])/xSize)
		pW2 *= abs((temp_xw[xSize]-temp_xw[0])/xSize)
	else
		pL1 = DimOffset(temp_yw,0) + pL1*DimDelta(temp_yw,0)
		pL2 = DimOffset(temp_yw,0) + pL2*DimDelta(temp_yw,0)
		pW1 *= abs(DimDelta(temp_yw,0))
		pW2 *= abs(DimDelta(temp_yw,0))
	endif
	
	WaveStats/Q temp_yw
	pH1 = (!pH1 || numtype(pH1)) ? (V_max/20) : pH1
	pH2 = (!pH2 || numtype(pH2)) ? (V_max/20) : pH2
	Make/D/O/N=7 W_coef = {wMin, pH1, pH2, pL1, pL2, pW1, pW2}
	WAVE s.cw = W_coef
	
	return 0	// no error
End

Function/S DoubleGauss_derivedVals(STRUCT SuperQuickFitStruct &s)

	variable sqrtLn	= 2*sqrt(2*ln(2))
	variable FWHM1	= sqrtLn * abs(s.cw[5])
	variable FWHM2	= sqrtLn * abs(s.cw[6])
	variable F1_err	= sqrtLn * sqrt(s.sw[5][5])
	variable F2_err	= sqrtLn * sqrt(s.sw[6][6])
	variable Amp1	= s.cw[1]/(abs(s.cw[5]) * sqrt(2*Pi))
	variable Amp2	= s.cw[2]/(abs(s.cw[6]) * sqrt(2*Pi))
	variable A1_err	= Amp1 * sqrt( (s.sw[5][5]/s.cw[5]^2) + (s.sw[1][1]/s.cw[1]^2) + 2*s.sw[1][5]/(s.cw[1]*s.cw[5]) )
	variable A2_err	= Amp2 * sqrt( (s.sw[6][6]/s.cw[6]^2) + (s.sw[1][1]/s.cw[1]^2) + 2*s.sw[1][6]/(s.cw[1]*s.cw[6]) )
	
	string str;	sPrintf str, "Derived values:\r\tAmp1\t= %g ± %g\r\tAmp2\t= %g ± %g\r\tFWHM1\t= %g ± %g\r\tFWHM2\t= %g ± %g", Amp1, A1_err, Amp2, A2_err, FWHM1, F1_err, FWHM2, F2_err
	if (s.doTextbox)
		AppendText "\t" + str
	endif
	
	return str
End

// *** helper code for fit functions ***############################################################

static Function/WAVE doFindPeaks(WAVE w, int p0, int p1)											// finds a list of possible peaks between points p0 and p1

	// below code is a condensed version of the Peak AutoFind code of the official Multipeak Fit package
	variable resampleFactor = ((p1-p0) < 150) ? max(2,ceil(150/(p1-p0))) : 1						// data fidelity might be too small
	Duplicate/FREE w, waveResampled
	Resample/UP=(resampleFactor)/WINF=None waveResampled
	SetScale/P x,0,1,waveResampled
	p0 *= resampleFactor
	p1 *= resampleFactor
	
	// ++++++++++++++++++++++++++++++ estimate the smoothing factor ++++++++++++++++++++++++++++++++
	Duplicate/FREE waveResampled, waveDiff; 	Differentiate waveDiff
	Duplicate/FREE waveDiff, waveDiff_copy
	
	variable maxVal = round(max(2,(p1-p0+1)/20))
	variable hSize  = 1000
	variable linRng = 10
	variable sqrRng = 20
	
	int smthFact = limit((p1-p0)/50,2,8)															// smoothing factor scales with size
	if (maxVal > 19)																				// introduce spacing if the no. of steps gets too large
		Make/FREE/N=(linRng + sqrRng) smoothed_SNratio = 0, smoothFactor = p+1
		smoothFactor[linRng,*]= ceil(linRng + (maxVal-linRng)*( (p+1-linRng)/sqrRng )^2)			// increase smooth steps to speed up the process in the latter part
	else
		Make/FREE/N=(maxVal) smoothed_SNratio = 0, smoothFactor = p+1
	endif
	
	int i = 0
	Make/FREE/N=(hSize--) DataHistogram
	do
		Duplicate/FREE waveDiff_copy, waveDiff;		Smooth/E=2/B=3 2*smoothFactor[i]+1, waveDiff
		Histogram/B=1 waveDiff, DataHistogram;		Integrate DataHistogram
		FindLevel/Q DataHistogram, (0.5-0.1)*DataHistogram[hSize]; variable x0 = V_LevelX
		FindLevel/Q DataHistogram, (0.5+0.1)*DataHistogram[hSize]; variable x1 = V_LevelX
		smoothed_SNratio[i++]= (pnt2x(DataHistogram,hSize)-pnt2x(DataHistogram,0))/(x1-x0)			// how much has the smoothing washed out the signal?
	while(i<numpnts(smoothFactor))
	WaveTransform zapNaNs smoothed_SNratio
	WaveStats/Q/R=[2,] smoothed_SNratio
	if (V_maxloc > -1)
		smthFact = smoothFactor[V_maxloc]
	endif
	
	// ++++++++++++++++++++++++++++++++ find peaks in the data +++++++++++++++++++++++++++++++++++++
	Duplicate/FREE/R=[p0,p1] waveResampled, waveSmoothed, waveDiff; Smooth/E=2/B=3   smthFact, waveDiff, waveSmoothed
	Differentiate waveDiff;					 						Smooth/E=2/B=3 2*smthFact, waveDiff
	Differentiate waveDiff;											Smooth/E=2/B=3 2*smthFact, waveDiff

	Make/D/N=(0,4)/FREE out																			// (values in points): [0] location, [1] width, [2] height, [3] strongest change
	int pS = 0
	int wn = DimSize(waveDiff,0)-1
	do
		FindPeak/B=1/N/P/R=[pS,]/Q waveDiff; variable pC = V_PeakLoc; variable maxC = V_PeakVal		// peak center
		if (V_flag)
			break
		endif
		FindPeak/B=1/P/R=[pC,0]/Q waveDiff; variable pL=V_flag ? 0  : V_PeakLoc; variable maxL=V_flag ? waveDiff[0]  : V_PeakVal
		FindPeak/B=1/P/R=[pC, ]/Q waveDiff; variable pR=V_flag ? wn : V_PeakLoc; variable maxR=V_flag ? waveDiff[pR] : V_PeakVal
		variable height = 1.3*(waveSmoothed[pC]-min(waveSmoothed[pR],waveSmoothed[pL]))				// approx. height from distance between center and edge
		out[DimSize(out,0)][] = {{pC+p0}, {abs(pR-pL)/sqrt(6)}, {height}, {min(abs(maxC-maxL),abs(maxC-maxR))}}
		if (V_flag)
			break
		endif
		pS = pR
	while(1)
	if (DimSize(out,0) > 0)																			// sort from strongest to weakest peak
		Make/D/N=(DimSize(out,0))/FREE heightSortWave = out[p][3]
		MakeIndex/R heightSortWave, heightSortWave
		Duplicate/FREE out, peakResultsCP
		out = peakResultsCP[heightSortWave[p]][q]
	endif
	
	out[][0,1] /= resampleFactor
	return out
End

// #################################################################################################

static Function getShapeFromData(STRUCT SuperQuickFitStruct &s,variable &loc, variable &amp, variable &base, variable &lwidth, variable &rwidth)	// finds basic parameters like height, width and location from input data

	WAVE   temp_yw = SQF_getTraceWave(s)
	WAVE/Z temp_xw = SQF_getTraceXWave(s)
	WaveStats/Q temp_yw
	amp  = (abs(V_max)>abs(V_min)) ? (V_max-V_min) : (V_min-V_max)
	base = (s.holdMode == 1) ? 0 : s.yBase
	if (WaveExists(temp_xw))
		loc = (amp > 0) ? temp_xw[V_maxRowloc] : temp_xw[V_minRowloc]
	else
		loc = (amp > 0) ? V_maxloc : V_minloc
	endif
	lwidth = NaN; rwidth = NaN
	
	// +++++++++++++++++++++++++++++++++ width and asymmetry +++++++++++++++++++++++++++++++++++++++
	Make/FREE/D/N=2 Levels
	variable rev
	variable avg = ceil((s.pntMax-s.pntMin)*0.05)
	if(WaveExists(temp_xw))
		int xSize = DimSize(temp_xw,0)-1
		rev = sign(temp_xw[xSize]-temp_xw[0])
		FindLevels/Q/D=Levels/B=(limit(avg,1,20))/P temp_yw, amp/2+base								// find the half-maximum positions
		if (V_LevelsFound == 2)
			if (Levels[0] > 0 && Levels[1] > 0 && Levels[0] < xSize && Levels[1] < xSize)
				lwidth = (loc - temp_xw[Levels[0]])*rev
				rwidth = (temp_xw[Levels[1]] - loc)*rev
			endif
		endif
	else
		rev = sign(DimDelta(s.data,s.yw_trDim))
		FindLevels/Q/D=Levels/B=(limit(avg,1,20))   temp_yw, amp/2+base
		if (V_LevelsFound == 2)
			lwidth = (loc - Levels[0])*rev
			rwidth = (Levels[1] - loc)*rev
		endif
	endif
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

static Function getPeakParameters(string PeakFunc, WAVE cw, variable &amp, variable &loc, variable &fwhm)	// outputs peak height, location and FWHM from a set of initial guesses

	variable isPos = amp > 0
	Optimize/Q/A=(isPos)/L=(loc-5*fwhm)/H=(loc+5*fwhm) $PeakFunc,cw									// find peak maximum numerically
	amp  = isPos ? V_max : V_min
	loc  = isPos ? V_maxLoc : V_minLoc
	fwhm = SuperQuickFit#getFWHM(PeakFunc,cw, amp, loc, fwhm)
	
	return 0
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

static Function getFWHM(string PeakFunc, WAVE cw, variable amp, variable loc, variable start_w)		// finds the approximate FWHM numerically

	variable hRange, lRange, highHW, LowHW
	hRange = (amp > 0) ? loc : (loc+5*start_w)
	lRange = (amp > 0) ? (loc+5*start_w) : loc
	FindRoots/Q/H=(hRange)/L=(lRange)/T=1e-15/Z=(amp/2) $PeakFunc,cw
	highHW = V_flag ? NaN : V_Root
	
	hRange = (amp > 0) ? loc : (loc-5*start_w)
	lRange = (amp > 0) ? (loc-5*start_w) : loc
	FindRoots/Q/H=(hRange)/L=(lRange)/T=1e-15/Z=(amp/2) $PeakFunc,cw
	LowHW  = V_flag ? NaN : V_Root
	
	return abs(highHW - LowHW)
End

// *** EMG Peak function definition *** ############################################################

Function ExpModGauss(WAVE cw, WAVE yw, WAVE xw)	: FitFunc

	Make/FREE/D/N=4 peak_cw = cw[p]		// 5 coeff. in cw: [0]: x0, [1]: width, [2]: amp, [3]: asym, [4]: y0
	variable dummy = MPFXEMGPeak(peak_cw, yw, xw)
	yw += cw[4]
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

Function ExpModGauss_prepareCoef(STRUCT SuperQuickFitStruct &s)										// the coef generator

	Make/D/O/N=5 W_coef	= NaN			// [0]: x0, [1]: width, [2]: amplitude, [3]: asymmetry, [4]: y0
	variable, loc, amp, base, lwidth, rwidth
	SuperQuickFit#getShapeFromData(s, loc, amp, base, lwidth, rwidth)								// gets location, amplitude, baseline and width from the data
	
	W_coef[0] = loc
	W_coef[2] = amp
	W_coef[4] = base
	if (!numtype(lwidth + rwidth))																	// half-maximum positions found
		W_coef[0] += (lwidth - rwidth)/2															// move the peak over a bit
		if (lwidth > rwidth)																		// left tail is longer
			W_coef[1] = rwidth
			W_coef[3] = -3*abs(lwidth - rwidth)														// updated ExpTau calculation to work well with asymmetric peak edit
			W_coef[3] = min(W_coef[3],-0.2*W_coef[1])												// ExpModGauss needs some minimal asymmetry
		else
			W_coef[1] = lwidth
			W_coef[3] = 3*abs(lwidth - rwidth)
			W_coef[3] = max(W_coef[3],0.2*W_coef[1])
		endif
		variable widthRatio = abs(W_coef[3])/W_coef[1]
		W_coef[2] = W_coef[2]/(1-(widthRatio/(widthRatio+1.2))^2)									// totally empirically determined
	else
		W_coef[1] = 1																				// may not be good guesses
		W_coef[3] = 0.2
	endif
	
	WAVE s.cw = W_coef
	s.holdStr = "00001"																				// may hold baseline
	
	return 0	// no error
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

Function/S ExpModGauss_derivedVals(STRUCT SuperQuickFitStruct &s)									// calculate derived parameters

	variable loc  = 	s.cw[0]																		// must provide (rough) guesses
	variable fwhm = abs(s.cw[1])
	variable amp  = 	s.cw[2]
	SuperQuickFit#getPeakParameters("SuperQuickFit#EMGFunc", s.cw, amp, loc, fwhm)
	
	variable pArea     = s.cw[1]*s.cw[2]*sqrt(2*pi)													// the peak area can be calculated analytically
	variable pArea_err = sqrt(s.sw[1][1]*(s.cw[2]*sqrt(2*pi))^2 + s.sw[2][2]*(s.cw[1]*sqrt(2*pi))^2 + 2*abs(s.sw[1][2])*(s.cw[1]*s.cw[2]*2*pi))
	
	string str;	sPrintf str, "Area\t= %.5g ± %.5g\r\tAmp\t= %.7g\r\tLoc\t= %.7g\r\tFWHM\t= %.7g", pArea, pArea_err, amp, loc, fwhm
	if (s.doTextbox)
		AppendText "\t" + str																		// also add derived parameters into text-box
	endif
	
	Print "K0 = x0, K1 = width, K2 = amplitude, K3 = asymmetry, K4 = y0"							// print explanation
	return "Derived values:\r\t" + str
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

static Function EMGFunc(WAVE w, variable x)															// for finding peak values

	Make/D/FREE/N=1 ytemp, xtemp=x
	variable dummy = MPFXEMGPeak(w, ytemp, xtemp)
	return ytemp[0]
End

// *** ExpConvExp function definition *** ##########################################################

Function ExpConvExp(WAVE cw, WAVE yw, WAVE xw)														// the fit func

	Make/FREE/D/N=4 peak_cw = cw[p]
	int dim = DimSize(xw,0)-1
	if (xw[0] > xw[dim])																			// for inverted x axes
		Duplicate/FREE xw, temp_xw
		xw = temp_xw[dim-p]
	endif
	variable dummy = MPFXExpConvExpPeak(peak_cw, yw, xw)
	yw += cw[4]
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

Function ExpConvExp_prepareCoef(STRUCT SuperQuickFitStruct &s)										// the coef generator

	Make/D/O/N=5 W_coef	= NaN	// [0]: x0, [1]: height, [2]: 1st inverse decay constant, [3]: 2nd inverse decay constant, [4]: y0
	variable, loc, amp, base, lwidth, rwidth
	SuperQuickFit#getShapeFromData(s, loc, amp, base, lwidth, rwidth)
	
	W_coef[0] = loc
	W_coef[1] = amp
	W_coef[4] = base
	if (!numtype(lwidth + rwidth))																	// half-maximum positions found
		W_coef[2] = 1/lwidth
		W_coef[3] = 1/rwidth
	else
		W_coef[2] = 1/(0.04*(s.pntMax-s.pntMin)*abs(DimDelta(s.data,s.yw_trDim)))					// may not be good guesses
		W_coef[3] = 1/(0.06*(s.pntMax-s.pntMin)*abs(DimDelta(s.data,s.yw_trDim)))
	endif
	if (W_coef[2] >= W_coef[3])																		// coefficients should not be exactly the same
		W_coef[3] = 1.2*W_coef[2]																	// make sure that decay coefficients are in the right order
	endif
	W_coef[1] *= 1.6																				// make the peak a bit higher
	W_coef[0] -= (ln(W_coef[2]/W_coef[3])/(W_coef[2]-W_coef[3]))									// shift the peak over
	
	WAVE s.cw = W_coef
	s.holdStr = "00001"																				// may hold baseline
	
	return 0	// no error
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

Function/S ExpConvExp_derivedVals(STRUCT SuperQuickFitStruct &s)									// calculate derived parameters

	variable amp, loc, fwhm
	SuperQuickFit#getExpConvExpVals(s.cw, amp, loc, fwhm)

	variable pArea = s.cw[1]/s.cw[3]																// the peak area can be calculated analytically
	variable pArea_err = sqrt(s.sw[1][1]/(s.cw[3])^2 + s.sw[3][3]*(s.cw[1]/s.cw[3]^2)^2 + 2*abs(s.sw[1][3])*(s.cw[1]/s.cw[3]^3))

	variable K3 = s.cw[2]
	variable K3_err = sqrt(s.sw[2][2])
	variable K2_mod = s.cw[3]-s.cw[2]																// add alternative coefficients
	variable K2_err = sqrt(s.sw[2][2] + s.sw[3][3] + 2*abs(s.sw[2][3]))
	variable K1_mod = (s.cw[2]*s.cw[1])/K2_mod
	
	variable d1 = (s.cw[2])/K2_mod																	// derivatives
	variable d2 = (s.cw[3]*s.cw[1])/K2_mod^2
	variable d3 = (s.cw[2]*s.cw[1])/K2_mod^2
	variable K1_err = s.sw[1][1]*(d1)^2 + s.sw[2][2]*(d2)^2 + s.sw[3][3]*(d3)^2
	K1_err += 2*abs(s.sw[1][3])*d1*d3 + 2*abs(s.sw[1][2])*d1*d2 + 2*abs(s.sw[2][3])*d2*d3			// side terms
	K1_err = sqrt(K1_err)

	string invStr = ""
	// +++++++++++++++++++++++++++++++++++
	WAVE/Z temp_xw = SQF_getTraceXWave(s)
	variable isRev = DimDelta( s.data,0) < 0
	if (WaveExists(temp_xw))
		isRev = temp_xw[0] > temp_xw[DimSize(temp_xw,0)-1]
	endif
	if (isRev)																						// invert result
		variable left  = DimOffset(s.fit,0)
		variable right = DimOffset(s.fit,0) + DimDelta(s.fit,0)*(DimSize(s.fit,0)-1)
		sPrintf invStr, "\tData inverted!\r\tCorrected K0 = %.7g\r\t", (right + left - s.cw[0])
		loc = right + left - loc
		SetScale/I x, right, left, WaveUnits(s.fit,0), s.fit
	endif
	// +++++++++++++++++++++++++++++++++++
	
	string baseStr, advStr
	sPrintf baseStr, "%sArea\t= %.5g ± %.5g\r\tHeight\t= %.7g\r\tLocation\t= %.7g\r\tFWHM\t= %.7g", invStr, pArea, pArea_err, amp, loc, fwhm
	sPrintf advStr,  "Virt. amplitude\t= %.5g ± %.5g\r\tInv. of rise\t= %.5g ± %.5g\r\tInv. of decay\t= %.5g ± %.5g", K1_mod, K1_err, K2_mod, K2_err, K3, K3_err
	if (s.doTextbox)
		AppendText "\t" + baseStr + "\r\t-----\r\t" + advStr										// also add derived parameters into text-box
	endif
	
	Print "K0 = location, K1 = amplitude,  K2 = 1st inv. decay const., K3 = 2nd inv. decay const., K4 = baseline"	// print explanation
	return "Derived values:\r\t" + baseStr + "\r\t-----\r\t" + advStr	
End

// ++++++++++++++++++++++++++++++ recreated from Multipeak Fit +++++++++++++++++++++++++++++++++++++

static Function getExpConvExpVals(WAVE cw, variable &amp, variable &loc, variable &fwhm)

	if (cw[2] == cw[3])
		amp = cw[1]/e
		loc = 1/cw[3] + cw[0]
	else
		amp = (cw[1]*cw[2] * ((cw[2]/cw[3])^(-(cw[2]/(cw[2]-cw[3]))) - (cw[2]/cw[3])^(-(cw[3]/(cw[2] - cw[3]))))) / (-cw[2] + cw[3])
		loc = ln(cw[2]/cw[3])/(cw[2]-cw[3]) + cw[0]
	endif
	fwhm = SuperQuickFit#getFWHM("SuperQuickFit#ExpConvExpFunc",cw, amp, loc, abs(1/cw[2] + 1/cw[3]))
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

static Function ExpConvExpFunc(WAVE w, variable x)													// for finding peak values

	Make/D/FREE/N=1 ytemp, xtemp=x
	variable dummy = MPFXExpConvExpPeak(w, ytemp, xtemp)
	return ytemp[0]
End

// *** Wave1D data fit function definition *** #####################################################

static StrConstant kWave1DPathStorage = "root:SQF_Wave1D_source"

Function Use_Wave_Data(WAVE w, variable x) : FitFunc

	//CurveFitDialog/ Equation:
	//CurveFitDialog/ f(x) = A*data(x-x0) + y0
	//CurveFitDialog/ End of Equation
	//CurveFitDialog/ Independent Variables 1
	//CurveFitDialog/ x
	//CurveFitDialog/ Coefficients 3
	//CurveFitDialog/ w[0] = y0
	//CurveFitDialog/ w[1] = A
	//CurveFitDialog/ w[2] = x0
	
	WAVE/Z data = $StrVarOrDefault(kWave1DPathStorage, "")
	if (!WaveExists(data))
		return 0
	endif
	if ((WaveType(data,1) == 2) || WaveDims(data) > 1)												// no text or multidim data here
		return 0
	endif
	
	variable pnt = limit((x - w[2] - DimOffset(data,0)) / DimDelta(data,0), 0, DimSize(data,0)-1)
	return w[1]*data[pnt] + w[0]
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

Function Use_Wave_Data_prepareCoef(STRUCT SuperQuickFitStruct &s)

	string path = StrVarOrDefault(kWave1DPathStorage, "")
	Prompt path, "Set the FULL path to the source 1D data:"
	DoPrompt "Setting up the data for fitting", path
	if (V_flag)
		return -1
	endif
	
	String/G $kWave1DPathStorage = path
	Print "•String/G " + kWave1DPathStorage + " = \"" + path + "\""
	
	WAVE tempYw = SQF_getTraceWave(s)
	WAVE/Z data = $StrVarOrDefault(kWave1DPathStorage, "")
	if (!WaveExists(data))
		return -1
	endif
	if ((WaveType(data,1) == 2) || WaveDims(data) > 1)												// no text or multidim data here
		return 0
	endif
	
	variable maxVal, minVal, maxPos
	WaveStats/Q tempYw
	maxPos = (!numtype(V_maxloc)) ? V_maxloc : 1
	maxVal = (!numtype(V_max) && V_max) ? V_max : 1
//	minVal = (!numtype(V_min) && V_min) ? V_min : 0
	
	WaveStats/Q data
	maxPos = (!numtype(V_maxloc)) ? (maxPos-V_maxloc) : 0
	maxVal = (!numtype(V_max) && V_max) ? (maxVal/V_max) : 1
	minVal = (!numtype(V_min)) ? (s.yBase-V_min) : s.yBase
	minVal = (s.holdMode == 1) ? 0 : minVal
	
	Make/D/O/N=2 W_coef = {minVal, maxVal, maxPos}
	WAVE s.cw = W_coef
	return 0	// no error
End

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

Function/S Use_Wave_Data_derivedVals(STRUCT SuperQuickFitStruct &s)

	string path = StrVarOrDefault(kWave1DPathStorage, "")
	WAVE/Z data = $path
	if (!WaveExists(data))
		return ""
	endif
	
	if (s.doTextbox)
		AppendText "\tUsed Data:\r\t" + NameOfWave(data)
	endif	
	return "Used Data: " + path
End
